<?php
session_start();
require_once 'config/database.php';
require_once 'classes/Invoice.php';
require_once 'classes/Customer.php';

$database = new Database();
$db = $database->getConnection();

// Check if database connection exists
if (!$db) {
    die("Database connection failed. Please check your configuration.");
}

// Initialize variables with default values
$total_revenue = 0;
$total_invoices = 0;
$customer_count = 0;
$pending_invoices = 0;

$invoice = new Invoice($db);
$customer = new Customer($db);

// Get recent invoices with error handling
try {
    $recent_invoices = $invoice->readAll(10, 0);
} catch (Exception $e) {
    $recent_invoices = null;
    $error_message = "Error loading recent invoices: " . $e->getMessage();
}

// Get customer count with error handling
try {
    $customer_stmt = $customer->readAll();
    $customer_count = $customer_stmt->rowCount();
} catch (Exception $e) {
    $customer_count = 0;
}

// Calculate total revenue with error handling
try {
    $revenue_query = "SELECT SUM(total_amount) as total_revenue, COUNT(*) as total_invoices FROM invoices WHERE status != 'cancelled'";
    $revenue_stmt = $db->prepare($revenue_query);
    $revenue_stmt->execute();
    $revenue_row = $revenue_stmt->fetch(PDO::FETCH_ASSOC);
    $total_revenue = $revenue_row['total_revenue'] ?? 0;
    $total_invoices = $revenue_row['total_invoices'] ?? 0;
} catch (Exception $e) {
    $total_revenue = 0;
    $total_invoices = 0;
}

// Get pending invoices count
try {
    $pending_query = "SELECT COUNT(*) as pending_count FROM invoices WHERE status IN ('draft', 'sent') AND (due_date < CURDATE() OR due_date IS NULL)";
    $pending_stmt = $db->prepare($pending_query);
    $pending_stmt->execute();
    $pending_row = $pending_stmt->fetch(PDO::FETCH_ASSOC);
    $pending_invoices = $pending_row['pending_count'] ?? 0;
} catch (Exception $e) {
    $pending_invoices = 0;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice Dashboard - ELECTROFIX SOLUTION</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .dashboard-card {
            border-radius: 15px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease;
        }
        .dashboard-card:hover {
            transform: translateY(-5px);
        }
        .stat-icon {
            font-size: 2.5rem;
            opacity: 0.8;
        }
        .sidebar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .nav-link {
            color: rgba(255, 255, 255, 0.8);
            transition: all 0.3s ease;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar p-0">
                <div class="p-4">
                    <h4 class="text-white mb-4">
                        <i class="fas fa-bolt"></i> ELECTROFIX
                    </h4>
                    <nav class="nav flex-column">
                        <a class="nav-link active mb-2" href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-2"></i> Dashboard
                        </a>
                        <a class="nav-link mb-2" href="index.php">
                            <i class="fas fa-plus me-2"></i> New Invoice
                        </a>
                        <a class="nav-link mb-2" href="invoices.php">
                            <i class="fas fa-file-invoice me-2"></i> All Invoices
                        </a>
                        <a class="nav-link mb-2" href="customers.php">
                            <i class="fas fa-users me-2"></i> Customers
                        </a>
                        <a class="nav-link mb-2" href="products.php">
                            <i class="fas fa-box me-2"></i> Products
                        </a>
                        <a class="nav-link mb-2" href="reports.php">
                            <i class="fas fa-chart-bar me-2"></i> Reports
                        </a>
                        <a class="nav-link mb-2" href="settings.php">
                            <i class="fas fa-cog me-2"></i> Settings
                        </a>
                    </nav>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 col-lg-10">
                <div class="p-4">
                    <!-- Header -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h2>Dashboard</h2>
                        <div>
                            <a href="index.php" class="btn btn-primary">
                                <i class="fas fa-plus"></i> New Invoice
                            </a>
                        </div>
                    </div>

                    <!-- Stats Cards -->
                    <div class="row mb-4">
                        <div class="col-md-3 mb-3">
                            <div class="card dashboard-card bg-primary text-white">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="card-title">Total Invoices</h6>
                                            <h3><?= number_format($total_invoices) ?></h3>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="fas fa-file-invoice"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-3">
                            <div class="card dashboard-card bg-success text-white">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="card-title">Total Revenue</h6>
                                            <h3>₹<?= number_format($total_revenue, 2) ?></h3>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="fas fa-rupee-sign"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-3">
                            <div class="card dashboard-card bg-info text-white">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="card-title">Total Customers</h6>
                                            <h3><?= number_format($customer_count) ?></h3>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="fas fa-users"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3 mb-3">
                            <div class="card dashboard-card bg-warning text-white">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h6 class="card-title">Pending Invoices</h6>
                                            <h3><?= number_format($pending_invoices) ?></h3>
                                            <small>Overdue or Draft</small>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="fas fa-clock"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Recent Invoices -->
                    <div class="card dashboard-card">
                        <div class="card-header">
                            <h5 class="mb-0">Recent Invoices</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Invoice No</th>
                                            <th>Customer</th>
                                            <th>Date</th>
                                            <th>Amount</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if ($recent_invoices && $recent_invoices->rowCount() > 0): ?>
                                            <?php while ($row = $recent_invoices->fetch(PDO::FETCH_ASSOC)): ?>
                                            <tr>
                                                <td>
                                                    <a href="view_invoice.php?id=<?= $row['id'] ?>" class="text-decoration-none">
                                                        <?= htmlspecialchars($row['invoice_number']) ?>
                                                    </a>
                                                </td>
                                                <td><?= htmlspecialchars($row['customer_name']) ?></td>
                                                <td><?= date('d/m/Y', strtotime($row['invoice_date'])) ?></td>
                                                <td>₹<?= number_format($row['total_amount'], 2) ?></td>
                                                <td>
                                                    <span class="badge bg-<?= $row['status'] == 'paid' ? 'success' : ($row['status'] == 'sent' ? 'warning' : ($row['status'] == 'cancelled' ? 'danger' : 'secondary')) ?>">
                                                        <?= ucfirst($row['status']) ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <a href="view_invoice.php?id=<?= $row['id'] ?>" class="btn btn-sm btn-outline-primary" title="View">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <a href="edit_invoice.php?id=<?= $row['id'] ?>" class="btn btn-sm btn-outline-secondary" title="Edit">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <button onclick="printInvoice(<?= $row['id'] ?>)" class="btn btn-sm btn-outline-info" title="Print">
                                                        <i class="fas fa-print"></i>
                                                    </button>
                                                </td>
                                            </tr>
                                            <?php endwhile; ?>
                                        <?php else: ?>
                                            <tr>
                                                <td colspan="6" class="text-center py-4">
                                                    <div class="text-muted">
                                                        <i class="fas fa-file-invoice fa-2x mb-2"></i>
                                                        <p>No invoices found. <a href="index.php">Create your first invoice</a></p>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
function printInvoice(invoiceId) {
    window.open(`view_invoice.php?id=${invoiceId}&print=1`, '_blank');
}

// Auto-refresh dashboard every 5 minutes
setInterval(function() {
    location.reload();
}, 300000);

// Add click handlers for stat cards
document.addEventListener('DOMContentLoaded', function() {
    // Make stat cards clickable
    document.querySelector('.bg-primary').addEventListener('click', function() {
        window.location.href = 'invoices.php';
    });
    
    document.querySelector('.bg-success').addEventListener('click', function() {
        window.location.href = 'reports.php';
    });
    
    document.querySelector('.bg-info').addEventListener('click', function() {
        window.location.href = 'customers.php';
    });
    
    document.querySelector('.bg-warning').addEventListener('click', function() {
        window.location.href = 'invoices.php?status=draft';
    });
});
</script>
</body>
</html>
