<?php
session_start();
require_once 'config/database.php';

$database = new Database();
$db = $database->getConnection();

// Date filters
$date_from = $_GET['date_from'] ?? date('Y-m-01'); // First day of current month
$date_to = $_GET['date_to'] ?? date('Y-m-d'); // Today

// Revenue by month
$revenue_query = "SELECT 
    DATE_FORMAT(invoice_date, '%Y-%m') as month,
    SUM(total_amount) as revenue,
    COUNT(*) as invoice_count
    FROM invoices 
    WHERE status != 'cancelled' 
    AND invoice_date BETWEEN ? AND ?
    GROUP BY DATE_FORMAT(invoice_date, '%Y-%m')
    ORDER BY month DESC";

$revenue_stmt = $db->prepare($revenue_query);
$revenue_stmt->execute([$date_from, $date_to]);

// Top customers
$customers_query = "SELECT 
    c.name,
    SUM(i.total_amount) as total_spent,
    COUNT(i.id) as invoice_count
    FROM customers c
    JOIN invoices i ON c.id = i.customer_id
    WHERE i.status != 'cancelled'
    AND i.invoice_date BETWEEN ? AND ?
    GROUP BY c.id, c.name
    ORDER BY total_spent DESC
    LIMIT 10";

$customers_stmt = $db->prepare($customers_query);
$customers_stmt->execute([$date_from, $date_to]);

// Invoice status summary
$status_query = "SELECT 
    status,
    COUNT(*) as count,
    SUM(total_amount) as amount
    FROM invoices 
    WHERE invoice_date BETWEEN ? AND ?
    GROUP BY status";

$status_stmt = $db->prepare($status_query);
$status_stmt->execute([$date_from, $date_to]);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports - ELECTROFIX SOLUTION</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .sidebar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .nav-link {
            color: rgba(255, 255, 255, 0.8);
            transition: all 0.3s ease;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 10px;
        }
        .report-card {
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        .chart-container {
            position: relative;
            height: 400px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar p-0">
                <div class="p-4">
                    <h4 class="text-white mb-4">
                        <i class="fas fa-bolt"></i> ELECTROFIX
                    </h4>
                    <nav class="nav flex-column">
                        <a class="nav-link mb-2" href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-2"></i> Dashboard
                        </a>
                        <a class="nav-link mb-2" href="index.php">
                            <i class="fas fa-plus me-2"></i> New Invoice
                        </a>
                        <a class="nav-link mb-2" href="invoices.php">
                            <i class="fas fa-file-invoice me-2"></i> All Invoices
                        </a>
                        <a class="nav-link mb-2" href="customers.php">
                            <i class="fas fa-users me-2"></i> Customers
                        </a>
                        <a class="nav-link mb-2" href="products.php">
                            <i class="fas fa-box me-2"></i> Products
                        </a>
                        <a class="nav-link active mb-2" href="reports.php">
                            <i class="fas fa-chart-bar me-2"></i> Reports
                        </a>
                        <a class="nav-link mb-2" href="settings.php">
                            <i class="fas fa-cog me-2"></i> Settings
                        </a>
                    </nav>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 col-lg-10">
                <div class="p-4">
                    <!-- Header -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <h2>Reports & Analytics</h2>
                            <p class="text-muted">Business insights and performance metrics</p>
                        </div>
                    </div>

                    <!-- Date Filter -->
                    <div class="card report-card">
                        <div class="card-body">
                            <form method="GET" class="row g-3">
                                <div class="col-md-4">
                                    <label class="form-label">From Date</label>
                                    <input type="date" class="form-control" name="date_from" value="<?= $date_from ?>">
                                </div>
                                <div class="col-md-4">
                                    <label class="form-label">To Date</label>
                                    <input type="date" class="form-control" name="date_to" value="<?= $date_to ?>">
                                </div>
                                <div class="col-md-4 d-flex align-items-end">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-filter"></i> Apply Filter
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Charts Row -->
                    <div class="row">
                        <div class="col-md-8">
                            <div class="card report-card">
                                <div class="card-header">
                                    <h5>Revenue Trend</h5>
                                </div>
                                <div class="card-body">
                                    <div class="chart-container">
                                        <canvas id="revenueChart"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card report-card">
                                <div class="card-header">
                                    <h5>Invoice Status</h5>
                                </div>
                                <div class="card-body">
                                    <div class="chart-container">
                                        <canvas id="statusChart"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Top Customers -->
                    <div class="card report-card">
                        <div class="card-header">
                            <h5>Top Customers</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Customer Name</th>
                                            <th>Total Spent</th>
                                            <th>Invoices</th>
                                            <th>Average Order</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if ($customers_stmt->rowCount() > 0): ?>
                                            <?php while ($row = $customers_stmt->fetch(PDO::FETCH_ASSOC)): ?>
                                                <tr>
                                                    <td><?= htmlspecialchars($row['name']) ?></td>
                                                    <td>₹<?= number_format($row['total_spent'], 2) ?></td>
                                                    <td><?= $row['invoice_count'] ?></td>
                                                    <td>₹<?= number_format($row['total_spent'] / $row['invoice_count'], 2) ?></td>
                                                </tr>
                                            <?php endwhile; ?>
                                        <?php else: ?>
                                            <tr>
                                                <td colspan="4" class="text-center">No data available for selected period</td>
                                            </tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Revenue Chart
        const revenueCtx = document.getElementById('revenueChart').getContext('2d');
        const revenueData = {
            labels: [
                <?php 
                $revenue_stmt->execute([$date_from, $date_to]);
                $labels = [];
                $amounts = [];
                while ($row = $revenue_stmt->fetch(PDO::FETCH_ASSOC)) {
                    $labels[] = "'" . date('M Y', strtotime($row['month'] . '-01')) . "'";
                    $amounts[] = $row['revenue'];
                }
                echo implode(',', array_reverse($labels));
                ?>
            ],
            datasets: [{
                label: 'Revenue (₹)',
                data: [<?= implode(',', array_reverse($amounts)) ?>],
                borderColor: 'rgb(75, 192, 192)',
                backgroundColor: 'rgba(75, 192, 192, 0.2)',
                tension: 0.1
            }]
        };

        new Chart(revenueCtx, {
            type: 'line',
            data: revenueData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '₹' + value.toLocaleString();
                            }
                        }
                    }
                }
            }
        });

        // Status Chart
        const statusCtx = document.getElementById('statusChart').getContext('2d');
        const statusData = {
            labels: [
                <?php 
                $status_stmt->execute([$date_from, $date_to]);
                $status_labels = [];
                $status_counts = [];
                $colors = [];
                $color_map = [
                    'draft' => '#6c757d',
                    'sent' => '#ffc107', 
                    'paid' => '#198754',
                    'cancelled' => '#dc3545'
                ];
                while ($row = $status_stmt->fetch(PDO::FETCH_ASSOC)) {
                    $status_labels[] = "'" . ucfirst($row['status']) . "'";
                    $status_counts[] = $row['count'];
                    $colors[] = "'" . ($color_map[$row['status']] ?? '#6c757d') . "'";
                }
                echo implode(',', $status_labels);
                ?>
            ],
            datasets: [{
                data: [<?= implode(',', $status_counts) ?>],
                backgroundColor: [<?= implode(',', $colors) ?>]
            }]
        };

        new Chart(statusCtx, {
            type: 'doughnut',
            data: statusData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    </script>
</body>
</html>
