// Enhanced JavaScript for invoice calculations and interactions

class InvoiceCalculator {
    constructor() {
        this.itemCount = 0;
        this.init();
    }

    init() {
        this.bindEvents();
        this.addInitialItem();
    }

    bindEvents() {
        // Add event listeners for dynamic calculations
        document.addEventListener('change', (e) => {
            if (e.target.classList.contains('item-qty') || e.target.classList.contains('item-rate')) {
                this.calculateItemTotal(e.target);
            }
        });

        // Tax type change
        const taxTypeSelect = document.getElementById('tax_type');
        if (taxTypeSelect) {
            taxTypeSelect.addEventListener('change', () => this.calculateTotals());
        }
    }

    addItem() {
        this.itemCount++;
        const container = document.getElementById('items-container');
        
        const itemHtml = `
            <div class="item-row" id="item-${this.itemCount}" data-item-id="${this.itemCount}">
                <div class="row">
                    <div class="col-md-4 mb-2">
                        <label class="form-label">Product Description</label>
                        <input type="text" class="form-control" name="items[${this.itemCount}][description]" 
                               placeholder="Enter product description" required>
                    </div>
                    <div class="col-md-2 mb-2">
                        <label class="form-label">HSN/SAC</label>
                        <input type="text" class="form-control" name="items[${this.itemCount}][hsn]" 
                               placeholder="HSN Code">
                    </div>
                    <div class="col-md-2 mb-2">
                        <label class="form-label">Quantity</label>
                        <input type="number" class="form-control item-qty" name="items[${this.itemCount}][qty]" 
                               min="1" step="1" placeholder="0" required>
                    </div>
                    <div class="col-md-2 mb-2">
                        <label class="form-label">Rate</label>
                        <input type="number" step="0.01" class="form-control item-rate" name="items[${this.itemCount}][rate]" 
                               min="0" placeholder="0.00" required>
                    </div>
                    <div class="col-md-1 mb-2">
                        <label class="form-label">Total</label>
                        <input type="text" class="form-control item-total" id="total-${this.itemCount}" 
                               readonly placeholder="0.00">
                    </div>
                    <div class="col-md-1 mb-2 d-flex align-items-end">
                        <button type="button" class="btn btn-outline-danger btn-sm remove-item" 
                                onclick="invoiceCalc.removeItem(${this.itemCount})" title="Remove Item">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        container.insertAdjacentHTML('beforeend', itemHtml);
        
        // Focus on the description field of the new item
        setTimeout(() => {
            const newDescInput = document.querySelector(`input[name="items[${this.itemCount}][description]"]`);
            if (newDescInput) newDescInput.focus();
        }, 100);
    }

    removeItem(itemId) {
        const itemElement = document.getElementById(`item-${itemId}`);
        if (itemElement) {
            // Add fade out animation
            itemElement.style.transition = 'all 0.3s ease';
            itemElement.style.opacity = '0';
            itemElement.style.transform = 'translateX(-100%)';
            
            setTimeout(() => {
                itemElement.remove();
                this.calculateTotals();
            }, 300);
        }
    }

    calculateItemTotal(input) {
        const itemRow = input.closest('.item-row');
        const itemId = itemRow.dataset.itemId;
        
        const qtyInput = itemRow.querySelector('.item-qty');
        const rateInput = itemRow.querySelector('.item-rate');
        const totalInput = itemRow.querySelector('.item-total');
        
        const qty = parseFloat(qtyInput.value) || 0;
        const rate = parseFloat(rateInput.value) || 0;
        const total = qty * rate;
        
        totalInput.value = this.formatCurrency(total);
        this.calculateTotals();
    }

    calculateTotals() {
        let subtotal = 0;
        const totalInputs = document.querySelectorAll('.item-total');
        
        totalInputs.forEach(input => {
            const value = input.value.replace(/[^\d.-]/g, ''); // Remove currency formatting
            subtotal += parseFloat(value) || 0;
        });
        
        const taxTypeSelect = document.getElementById('tax_type');
        const taxType = taxTypeSelect ? taxTypeSelect.value : 'cgst_sgst';
        
        let taxAmount = 0;
        if (taxType === 'cgst_sgst') {
            taxAmount = subtotal * 0.18; // 9% + 9%
        } else {
            taxAmount = subtotal * 0.18; // 18%
        }
        
        const totalAmount = subtotal + taxAmount;
        
        // Update display fields
        const subtotalField = document.getElementById('subtotal');
        const taxAmountField = document.getElementById('tax_amount');
        const totalAmountField = document.getElementById('total_amount');
        
        if (subtotalField) subtotalField.value = this.formatCurrency(subtotal);
        if (taxAmountField) taxAmountField.value = this.formatCurrency(taxAmount);
        if (totalAmountField) totalAmountField.value = this.formatCurrency(totalAmount);
    }

    formatCurrency(amount) {
        return new Intl.NumberFormat('en-IN', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        }).format(amount);
    }

    addInitialItem() {
        this.addItem();
    }

    // Validation functions
    validateForm() {
        const form = document.getElementById('invoiceForm');
        const requiredFields = form.querySelectorAll('[required]');
        let isValid = true;
        
        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                field.classList.add('is-invalid');
                isValid = false;
            } else {
                field.classList.remove('is-invalid');
            }
        });
        
        // Check if at least one item exists
        const items = document.querySelectorAll('.item-row');
        if (items.length === 0) {
            alert('Please add at least one item to the invoice.');
            isValid = false;
        }
        
        return isValid;
    }

    // Auto-save functionality
    autoSave() {
        const formData = new FormData(document.getElementById('invoiceForm'));
        const data = Object.fromEntries(formData);
        localStorage.setItem('invoice_draft', JSON.stringify(data));
    }

    loadDraft() {
        const draft = localStorage.getItem('invoice_draft');
        if (draft) {
            const data = JSON.parse(draft);
            // Populate form fields with draft data
            Object.keys(data).forEach(key => {
                const field = document.querySelector(`[name="${key}"]`);
                if (field) field.value = data[key];
            });
        }
    }
}

// Initialize calculator when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    window.invoiceCalc = new InvoiceCalculator();
    
    // Add form validation on submit
    const form = document.getElementById('invoiceForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            if (!invoiceCalc.validateForm()) {
                e.preventDefault();
                return false;
            }
        });
    }
    
    // Auto-save every 30 seconds
    setInterval(() => {
        if (window.invoiceCalc) {
            invoiceCalc.autoSave();
        }
    }, 30000);
});

// Global functions for backward compatibility
function addItem() {
    if (window.invoiceCalc) {
        window.invoiceCalc.addItem();
    }
}

function removeItem(itemId) {
    if (window.invoiceCalc) {
        window.invoiceCalc.removeItem(itemId);
    }
}

function calculateItemTotal(itemId) {
    if (window.invoiceCalc) {
        const input = document.querySelector(`input[name="items[${itemId}][qty]"]`) || 
                     document.querySelector(`input[name="items[${itemId}][rate]"]`);
        if (input) {
            window.invoiceCalc.calculateItemTotal(input);
        }
    }
}

function calculateTotals() {
    if (window.invoiceCalc) {
        window.invoiceCalc.calculateTotals();
    }
}
