<?php
session_start();
require_once 'config/database.php';
require_once 'classes/Invoice.php';
require_once 'classes/InvoiceItem.php';

// Check if we have an invoice ID from database
if (isset($_SESSION['invoice_id'])) {
    $database = new Database();
    $db = $database->getConnection();
    
    $invoice = new Invoice($db);
    $invoice_item = new InvoiceItem($db);
    
    // Get invoice from database
    $invoice->id = $_SESSION['invoice_id'];
    $invoice_data = $invoice->readOne();
    
    if ($invoice_data) {
        // Get invoice items
        $items_stmt = $invoice_item->readByInvoiceId($_SESSION['invoice_id']);
        $items = [];
        while ($item = $items_stmt->fetch(PDO::FETCH_ASSOC)) {
            $items[] = $item;
        }
        
        // Use database data
        $data = [
            'invoice_no' => $invoice_data['invoice_number'],
            'invoice_date' => $invoice_data['invoice_date'],
            'rgp_no' => $invoice_data['rgp_number'],
            'rgp_date' => $invoice_data['rgp_date'],
            'po_no' => $invoice_data['po_number'],
            'po_date' => $invoice_data['po_date'],
            'reverse_charge' => $invoice_data['reverse_charge'],
            'tax_type' => $invoice_data['tax_type'],
            'customer_name' => $invoice_data['customer_name'],
            'billing_address' => $invoice_data['billing_address'],
            'shipping_address' => $invoice_data['shipping_address'],
            'customer_gstin' => $invoice_data['customer_gstin'],
            'customer_state' => $invoice_data['customer_state'],
            'items' => $items
        ];
        
        $subtotal = $invoice_data['subtotal'];
        $tax_amount = $invoice_data['total_tax'];
        $total_amount = $invoice_data['total_amount'];
        
    } else {
        // Fallback to session data
        $data = $_SESSION['invoice_data'] ?? null;
        if (!$data) {
            header('Location: index.php');
            exit;
        }
        
        // Calculate totals from session data
        $subtotal = 0;
        if (isset($data['items'])) {
            foreach ($data['items'] as $item) {
                $subtotal += ($item['qty'] * $item['rate']);
            }
        }
        $tax_rate = 0.18;
        $tax_amount = $subtotal * $tax_rate;
        $total_amount = $subtotal + $tax_amount;
    }
    
    // Clear session after use
    unset($_SESSION['invoice_id']);
    
} else if (isset($_SESSION['invoice_data'])) {
    // Fallback to session data for backward compatibility
    $data = $_SESSION['invoice_data'];
    
    // Calculate totals
    $subtotal = 0;
    if (isset($data['items'])) {
        foreach ($data['items'] as $item) {
            $subtotal += ($item['qty'] * $item['rate']);
        }
    }
    
    $tax_rate = 0.18;
    $tax_amount = $subtotal * $tax_rate;
    $total_amount = $subtotal + $tax_amount;
} else {
    header('Location: index.php');
    exit;
}

// Company data
$company = [
    'name' => 'ELECTROFIX SOLUTION',
    'address' => 'D-171, 2nd Floor, Sector-10, Noida-201301',
    'contact' => '8851369694',
    'email' => 'contact.electrofixsolution@gmail.com',
    'gst' => '09DMFPP9986Q1ZJ',
    'state' => 'UTTAR PRADESH 09'
];

$bank = [
    'name' => 'HDFC Bank Ltd.',
    'account' => '50200086529101',
    'ifsc' => 'HDFC0000088'
];

// Number to words function
function numberToWords($number) {
    $ones = array(
        0 => 'ZERO', 1 => 'ONE', 2 => 'TWO', 3 => 'THREE', 4 => 'FOUR', 5 => 'FIVE',
        6 => 'SIX', 7 => 'SEVEN', 8 => 'EIGHT', 9 => 'NINE', 10 => 'TEN',
        11 => 'ELEVEN', 12 => 'TWELVE', 13 => 'THIRTEEN', 14 => 'FOURTEEN', 15 => 'FIFTEEN',
        16 => 'SIXTEEN', 17 => 'SEVENTEEN', 18 => 'EIGHTEEN', 19 => 'NINETEEN'
    );
    
    $tens = array(
        2 => 'TWENTY', 3 => 'THIRTY', 4 => 'FORTY', 5 => 'FIFTY',
        6 => 'SIXTY', 7 => 'SEVENTY', 8 => 'EIGHTY', 9 => 'NINETY'
    );
    
    if ($number < 20) {
        return $ones[$number];
    } elseif ($number < 100) {
        return $tens[intval($number / 10)] . ($number % 10 != 0 ? ' ' . $ones[$number % 10] : '');
    } elseif ($number < 1000) {
        return $ones[intval($number / 100)] . ' HUNDRED' . ($number % 100 != 0 ? ' ' . numberToWords($number % 100) : '');
    } elseif ($number < 100000) {
        return numberToWords(intval($number / 1000)) . ' THOUSAND' . ($number % 1000 != 0 ? ' ' . numberToWords($number % 1000) : '');
    } elseif ($number < 10000000) {
        return numberToWords(intval($number / 100000)) . ' LAKH' . ($number % 100000 != 0 ? ' ' . numberToWords($number % 100000) : '');
    } else {
        return numberToWords(intval($number / 10000000)) . ' CRORE' . ($number % 10000000 != 0 ? ' ' . numberToWords($number % 10000000) : '');
    }
}

$amount_in_words = 'INR ' . numberToWords(intval($total_amount)) . ' ONLY';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tax Invoice - <?= $data['invoice_no'] ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        @media print {
            .no-print { display: none !important; }
            body { font-size: 12px; }
            .invoice-container { box-shadow: none !important; }
        }
        
        .invoice-container {
            max-width: 210mm;
            margin: 0 auto;
            background: white;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            font-family: Arial, sans-serif;
        }
        
        .invoice-header {
            background: #f8f9fa;
            border-bottom: 3px solid #007bff;
            padding: 20px;
        }
        
        .company-name {
            font-size: 24px;
            font-weight: bold;
            color: #007bff;
            margin-bottom: 5px;
        }
        
        .invoice-title {
            background: #007bff;
            color: white;
            text-align: center;
            padding: 10px;
            font-size: 18px;
            font-weight: bold;
            margin: 20px 0;
        }
        
        .invoice-details {
            background: #f8f9fa;
            padding: 15px;
            border: 1px solid #dee2e6;
        }
        
        .address-section {
            border: 1px solid #dee2e6;
            padding: 15px;
            height: 100%;
        }
        
        .items-table {
            border-collapse: collapse;
            width: 100%;
            margin: 20px 0;
        }
        
        .items-table th,
        .items-table td {
            border: 1px solid #dee2e6;
            padding: 8px;
            text-align: left;
        }
        
        .items-table th {
            background: #f8f9fa;
            font-weight: bold;
            text-align: center;
        }
        
        .text-right {
            text-align: right;
        }
        
        .total-section {
            background: #f8f9fa;
            padding: 15px;
            border: 1px solid #dee2e6;
        }
        
        .signature-section {
            margin-top: 30px;
            text-align: right;
        }
        
        .terms-section {
            margin-top: 20px;
            padding: 15px;
            background: #f8f9fa;
            border: 1px solid #dee2e6;
        }
    </style>
</head>
<body>
    <div class="container-fluid p-4">
        <div class="no-print mb-3 text-center">
            <button onclick="window.print()" class="btn btn-primary me-2">
                <i class="fas fa-print"></i> Print Invoice
            </button>
            <a href="index.php" class="btn btn-secondary me-2">
                <i class="fas fa-arrow-left"></i> Back to Form
            </a>
            <a href="invoices.php" class="btn btn-success">
                <i class="fas fa-list"></i> View All Invoices
            </a>
        </div>

        <div class="invoice-container">
            <!-- Header -->
            <div class="invoice-header">
                <div class="row">
                    <div class="col-8">
                        <div class="company-name"><?= $company['name'] ?></div>
                        <div><?= $company['address'] ?></div>
                        <div>CONTACT: <?= $company['contact'] ?></div>
                        <div>Email: <?= $company['email'] ?></div>
                    </div>
                    <div class="col-4 text-end">
                        <!-- Logo space -->
                        <div style="width: 100px; height: 80px; border: 2px dashed #ccc; display: inline-block; text-align: center; line-height: 80px; color: #999;">
                            LOGO
                        </div>
                    </div>
                </div>
            </div>

            <!-- Invoice Title -->
            <div class="invoice-title">TAX INVOICE</div>

            <!-- Invoice Details -->
            <div class="row mb-3">
                <div class="col-6">
                    <div class="invoice-details">
                        <div><strong>INVOICE NO:</strong> <?= $data['invoice_no'] ?></div>
                        <div><strong>INVOICE DATE:</strong> <?= date('d/m/Y', strtotime($data['invoice_date'])) ?></div>
                        <div><strong>RGP NO:</strong> <?= $data['rgp_no'] ?></div>
                        <div><strong>RGP DATE:</strong> <?= date('d/m/Y', strtotime($data['rgp_date'])) ?></div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="invoice-details">
                        <div><strong>PO NO:</strong> <?= $data['po_no'] ?></div>
                        <div><strong>PO DATE:</strong> <?= date('d/m/Y', strtotime($data['po_date'])) ?></div>
                        <div><strong>GST NO:</strong> <?= $company['gst'] ?></div>
                        <div><strong>STATE:</strong> <?= $company['state'] ?></div>
                    </div>
                </div>
            </div>

            <div class="mb-3">
                <strong>Tax is payable on reverse charge:</strong> <?= $data['reverse_charge'] ?>
            </div>

            <!-- Addresses -->
            <div class="row mb-3">
                <div class="col-6">
                    <div class="address-section">
                        <h6><strong>Billed to:</strong></h6>
                        <div><strong>Name:</strong> <?= $data['customer_name'] ?></div>
                        <div><strong>Address:</strong> <?= $data['billing_address'] ?></div>
                        <div><strong>GSTIN:</strong> <?= $data['customer_gstin'] ?></div>
                        <div><strong>STATE:</strong> <?= $data['customer_state'] ?></div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="address-section">
                        <h6><strong>Shipped to:</strong></h6>
                        <div><strong>Name:</strong> <?= $data['customer_name'] ?></div>
                        <div><strong>Address:</strong> <?= $data['shipping_address'] ?></div>
                        <div><strong>GSTIN:</strong> <?= $data['customer_gstin'] ?></div>
                        <div><strong>STATE:</strong> <?= $data['customer_state'] ?></div>
                    </div>
                </div>
            </div>

            <!-- Items Table -->
            <table class="items-table">
                <thead>
                    <tr>
                        <th>S.NO.</th>
                        <th>Product Description/Service</th>
                        <th>HSN/SAC</th>
                        <th>QTY.</th>
                        <th>RATE/PCS</th>
                        <th>Total Amount</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (isset($data['items']) && is_array($data['items'])): ?>
                        <?php $sno = 1; ?>
                        <?php foreach ($data['items'] as $item): ?>
                            <tr>
                                <td class="text-center"><?= $sno++ ?></td>
                                <td><?= htmlspecialchars($item['description'] ?? '') ?></td>
                                <td class="text-center"><?= htmlspecialchars($item['hsn_sac'] ?? $item['hsn'] ?? '') ?></td>
                                <td class="text-center"><?= htmlspecialchars($item['quantity'] ?? $item['qty'] ?? '') ?></td>
                                <td class="text-right"><?= number_format($item['rate'], 2) ?></td>
                                <td class="text-right"><?= number_format(($item['quantity'] ?? $item['qty']) * $item['rate'], 2) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <!-- Totals Section -->
            <div class="row">
                <div class="col-6">
                    <div class="total-section">
                        <h6><strong>Bank Details:</strong></h6>
                        <div><strong>Bank Name:</strong> <?= $bank['name'] ?></div>
                        <div><strong>Bank Account Number:</strong> <?= $bank['account'] ?></div>
                        <div><strong>Bank IFSC:</strong> <?= $bank['ifsc'] ?></div>
                        
                        <div class="mt-3">
                            <strong>Total invoice amount in words:</strong><br>
                            <?= $amount_in_words ?>
                        </div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="total-section">
                        <div class="d-flex justify-content-between">
                            <span><strong>Total amount before tax:</strong></span>
                            <span><?= number_format($subtotal, 2) ?></span>
                        </div>
                        
                        <?php if ($data['tax_type'] === 'cgst_sgst'): ?>
                            <div class="d-flex justify-content-between">
                                <span>Add: CGST @9%</span>
                                <span><?= number_format($tax_amount / 2, 2) ?></span>
                            </div>
                            <div class="d-flex justify-content-between">
                                <span>Add: SGST @9%</span>
                                <span><?= number_format($tax_amount / 2, 2) ?></span>
                            </div>
                        <?php else: ?>
                            <div class="d-flex justify-content-between">
                                <span>Add: IGST @18%</span>
                                <span><?= number_format($tax_amount, 2) ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <hr>
                        <div class="d-flex justify-content-between">
                            <span><strong>Total tax amount:</strong></span>
                            <span><strong><?= number_format($tax_amount, 2) ?></strong></span>
                        </div>
                        <div class="d-flex justify-content-between">
                            <span><strong>Total amount after tax:</strong></span>
                            <span><strong><?= number_format($total_amount, 2) ?></strong></span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Terms and Signature -->
            <div class="terms-section">
                <div class="row">
                    <div class="col-6">
                        <h6><strong>PAYMENT TERMS & CONDITIONS</strong></h6>
                        <p>Net 10 - Payment ten days after the invoice date</p>
                        <p><strong>All Disputes Are Subject to Noida Jurisdiction</strong></p>
                    </div>
                    <div class="col-6">
                        <div class="signature-section">
                            <p><strong>For: <?= $company['name'] ?></strong></p>
                            <div style="height: 60px;"></div>
                            <p><strong>Authorized Signatory</strong></p>
                            <p><strong>E. & O.E.</strong></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/js/all.min.js"></script>
</body>
</html>
