<?php
session_start();
require_once 'config/database.php';
require_once 'classes/Invoice.php';
require_once 'classes/Customer.php';

$database = new Database();
$db = $database->getConnection();

$invoice = new Invoice($db);
$customer = new Customer($db);

// Handle search and filters
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$customer_filter = $_GET['customer_id'] ?? '';

// Pagination
$page = $_GET['page'] ?? 1;
$records_per_page = 20;
$offset = ($page - 1) * $records_per_page;

// Build query with filters
$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(i.invoice_number LIKE ? OR c.name LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($status_filter)) {
    $where_conditions[] = "i.status = ?";
    $params[] = $status_filter;
}

if (!empty($date_from)) {
    $where_conditions[] = "i.invoice_date >= ?";
    $params[] = $date_from;
}

if (!empty($date_to)) {
    $where_conditions[] = "i.invoice_date <= ?";
    $params[] = $date_to;
}

if (!empty($customer_filter)) {
    $where_conditions[] = "i.customer_id = ?";
    $params[] = $customer_filter;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total count for pagination
$count_query = "SELECT COUNT(*) as total FROM invoices i 
                LEFT JOIN customers c ON i.customer_id = c.id 
                $where_clause";
$count_stmt = $db->prepare($count_query);
foreach ($params as $index => $param) {
    $count_stmt->bindValue($index + 1, $param);
}
$count_stmt->execute();
$total_records = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];
$total_pages = ceil($total_records / $records_per_page);

// Get invoices with filters
$query = "SELECT i.*, c.name as customer_name, c.gstin as customer_gstin
          FROM invoices i
          LEFT JOIN customers c ON i.customer_id = c.id
          $where_clause
          ORDER BY i.created_at DESC
          LIMIT $records_per_page OFFSET $offset";

$stmt = $db->prepare($query);
foreach ($params as $index => $param) {
    $stmt->bindValue($index + 1, $param);
}
$stmt->execute();

// Get all customers for filter dropdown
$customers_stmt = $customer->readAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>All Invoices - ELECTROFIX SOLUTION</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <style>
        .sidebar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .nav-link {
            color: rgba(255, 255, 255, 0.8);
            transition: all 0.3s ease;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 10px;
        }
        .invoice-card {
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            transition: transform 0.3s ease;
        }
        .invoice-card:hover {
            transform: translateY(-2px);
        }
        .status-badge {
            font-size: 0.8rem;
            padding: 0.4rem 0.8rem;
            border-radius: 20px;
        }
        .filter-section {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .action-buttons .btn {
            margin: 2px;
        }
        .invoice-amount {
            font-weight: bold;
            color: #28a745;
        }
        .table-responsive {
            border-radius: 10px;
            overflow: hidden;
        }
        .pagination {
            justify-content: center;
        }
        .bulk-actions {
            background: #e9ecef;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 20px;
            display: none;
        }
        .invoice-row.selected {
            background-color: #e3f2fd;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar p-0">
                <div class="p-4">
                    <h4 class="text-white mb-4">
                        <i class="fas fa-bolt"></i> ELECTROFIX
                    </h4>
                    <nav class="nav flex-column">
                        <a class="nav-link mb-2" href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-2"></i> Dashboard
                        </a>
                        <a class="nav-link mb-2" href="index.php">
                            <i class="fas fa-plus me-2"></i> New Invoice
                        </a>
                        <a class="nav-link active mb-2" href="invoices.php">
                            <i class="fas fa-file-invoice me-2"></i> All Invoices
                        </a>
                        <a class="nav-link mb-2" href="customers.php">
                            <i class="fas fa-users me-2"></i> Customers
                        </a>
                        <a class="nav-link mb-2" href="products.php">
                            <i class="fas fa-box me-2"></i> Products
                        </a>
                        <a class="nav-link mb-2" href="reports.php">
                            <i class="fas fa-chart-bar me-2"></i> Reports
                        </a>
                        <a class="nav-link mb-2" href="settings.php">
                            <i class="fas fa-cog me-2"></i> Settings
                        </a>
                    </nav>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 col-lg-10">
                <div class="p-4">
                    <!-- Header -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <h2>All Invoices</h2>
                            <p class="text-muted">Manage and track all your invoices</p>
                        </div>
                        <div>
                            <a href="index.php" class="btn btn-primary">
                                <i class="fas fa-plus"></i> New Invoice
                            </a>
                            <button class="btn btn-outline-secondary" onclick="exportInvoices()">
                                <i class="fas fa-download"></i> Export
                            </button>
                        </div>
                    </div>

                    <!-- Filters Section -->
                    <div class="filter-section">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label class="form-label">Search</label>
                                <input type="text" class="form-control" name="search" 
                                       placeholder="Invoice number or customer..." 
                                       value="<?= htmlspecialchars($search) ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Status</label>
                                <select class="form-control" name="status">
                                    <option value="">All Status</option>
                                    <option value="draft" <?= $status_filter == 'draft' ? 'selected' : '' ?>>Draft</option>
                                    <option value="sent" <?= $status_filter == 'sent' ? 'selected' : '' ?>>Sent</option>
                                    <option value="paid" <?= $status_filter == 'paid' ? 'selected' : '' ?>>Paid</option>
                                    <option value="cancelled" <?= $status_filter == 'cancelled' ? 'selected' : '' ?>>Cancelled</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Customer</label>
                                <select class="form-control" name="customer_id">
                                    <option value="">All Customers</option>
                                    <?php while ($cust_row = $customers_stmt->fetch(PDO::FETCH_ASSOC)): ?>
                                        <option value="<?= $cust_row['id'] ?>" 
                                                <?= $customer_filter == $cust_row['id'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($cust_row['name']) ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">From Date</label>
                                <input type="date" class="form-control" name="date_from" 
                                       value="<?= htmlspecialchars($date_from) ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">To Date</label>
                                <input type="date" class="form-control" name="date_to" 
                                       value="<?= htmlspecialchars($date_to) ?>">
                            </div>
                            <div class="col-md-1 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary w-100">
                                    <i class="fas fa-search"></i>
                                </button>
                            </div>
                        </form>
                        <div class="mt-3">
                            <a href="invoices.php" class="btn btn-outline-secondary btn-sm">
                                <i class="fas fa-times"></i> Clear Filters
                            </a>
                            <span class="text-muted ms-3">
                                Showing <?= $total_records ?> invoice(s)
                            </span>
                        </div>
                    </div>

                    <!-- Bulk Actions -->
                    <div class="bulk-actions" id="bulkActions">
                        <div class="d-flex align-items-center">
                            <span class="me-3">
                                <span id="selectedCount">0</span> invoice(s) selected
                            </span>
                            <button class="btn btn-sm btn-outline-primary me-2" onclick="bulkUpdateStatus('sent')">
                                Mark as Sent
                            </button>
                            <button class="btn btn-sm btn-outline-success me-2" onclick="bulkUpdateStatus('paid')">
                                Mark as Paid
                            </button>
                            <button class="btn btn-sm btn-outline-danger me-2" onclick="bulkDelete()">
                                Delete Selected
                            </button>
                            <button class="btn btn-sm btn-outline-secondary" onclick="clearSelection()">
                                Clear Selection
                            </button>
                        </div>
                    </div>

                    <!-- Invoices Table -->
                    <div class="card invoice-card">
                        <div class="card-body p-0">
                            <div class="table-responsive">
                                <table class="table table-hover mb-0" id="invoicesTable">
                                    <thead class="table-light">
                                        <tr>
                                            <th width="40">
                                                <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                                            </th>
                                            <th>Invoice No</th>
                                            <th>Customer</th>
                                            <th>Date</th>
                                            <th>Due Date</th>
                                            <th>Amount</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if ($stmt->rowCount() > 0): ?>
                                            <?php while ($row = $stmt->fetch(PDO::FETCH_ASSOC)): ?>
                                                <tr class="invoice-row" data-invoice-id="<?= $row['id'] ?>">
                                                    <td>
                                                        <input type="checkbox" class="invoice-checkbox" 
                                                               value="<?= $row['id'] ?>" onchange="updateSelection()">
                                                    </td>
                                                    <td>
                                                        <strong><?= htmlspecialchars($row['invoice_number']) ?></strong>
                                                        <br>
                                                        <small class="text-muted">
                                                            Created: <?= date('d/m/Y H:i', strtotime($row['created_at'])) ?>
                                                        </small>
                                                    </td>
                                                    <td>
                                                        <div><?= htmlspecialchars($row['customer_name']) ?></div>
                                                        <?php if ($row['customer_gstin']): ?>
                                                            <small class="text-muted">GST: <?= htmlspecialchars($row['customer_gstin']) ?></small>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td><?= date('d/m/Y', strtotime($row['invoice_date'])) ?></td>
                                                    <td>
                                                        <?php if ($row['due_date']): ?>
                                                            <?= date('d/m/Y', strtotime($row['due_date'])) ?>
                                                            <?php
                                                            $due_date = new DateTime($row['due_date']);
                                                            $today = new DateTime();
                                                            $diff = $today->diff($due_date);
                                                            if ($today > $due_date && $row['status'] != 'paid'):
                                                            ?>
                                                                <br><small class="text-danger">
                                                                    <i class="fas fa-exclamation-triangle"></i> 
                                                                    Overdue by <?= $diff->days ?> days
                                                                </small>
                                                            <?php endif; ?>
                                                        <?php else: ?>
                                                            <span class="text-muted">-</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td class="invoice-amount">
                                                        ₹<?= number_format($row['total_amount'], 2) ?>
                                                        <br>
                                                        <small class="text-muted">
                                                            Tax: ₹<?= number_format($row['total_tax'], 2) ?>
                                                        </small>
                                                    </td>
                                                    <td>
                                                        <?php
                                                        $status_colors = [
                                                            'draft' => 'secondary',
                                                            'sent' => 'warning',
                                                            'paid' => 'success',
                                                            'cancelled' => 'danger'
                                                        ];
                                                        $status_color = $status_colors[$row['status']] ?? 'secondary';
                                                        ?>
                                                        <span class="badge bg-<?= $status_color ?> status-badge">
                                                            <?= ucfirst($row['status']) ?>
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <div class="action-buttons">
                                                            <a href="view_invoice.php?id=<?= $row['id'] ?>" 
                                                               class="btn btn-sm btn-outline-primary" title="View">
                                                                <i class="fas fa-eye"></i>
                                                            </a>
                                                            <a href="edit_invoice.php?id=<?= $row['id'] ?>" 
                                                               class="btn btn-sm btn-outline-secondary" title="Edit">
                                                                <i class="fas fa-edit"></i>
                                                            </a>
                                                            <button onclick="printInvoice(<?= $row['id'] ?>)" 
                                                                    class="btn btn-sm btn-outline-info" title="Print">
                                                                <i class="fas fa-print"></i>
                                                            </button>
                                                            <div class="btn-group">
                                                                <button class="btn btn-sm btn-outline-success dropdown-toggle" 
                                                                        type="button" data-bs-toggle="dropdown">
                                                                    <i class="fas fa-cog"></i>
                                                                </button>
                                                                <ul class="dropdown-menu">
                                                                    <li><a class="dropdown-item" href="#" 
                                                                           onclick="updateInvoiceStatus(<?= $row['id'] ?>, 'sent')">
                                                                        <i class="fas fa-paper-plane"></i> Mark as Sent
                                                                    </a></li>
                                                                    <li><a class="dropdown-item" href="#" 
                                                                           onclick="updateInvoiceStatus(<?= $row['id'] ?>, 'paid')">
                                                                        <i class="fas fa-check"></i> Mark as Paid
                                                                    </a></li>
                                                                    <li><a class="dropdown-item" href="#" 
                                                                           onclick="duplicateInvoice(<?= $row['id'] ?>)">
                                                                        <i class="fas fa-copy"></i> Duplicate
                                                                    </a></li>
                                                                    <li><hr class="dropdown-divider"></li>
                                                                    <li><a class="dropdown-item text-danger" href="#" 
                                                                           onclick="deleteInvoice(<?= $row['id'] ?>)">
                                                                        <i class="fas fa-trash"></i> Delete
                                                                    </a></li>
                                                                </ul>
                                                            </div>
                                                        </div>
                                                    </td>
                                                </tr>
                                            <?php endwhile; ?>
                                        <?php else: ?>
                                            <tr>
                                                <td colspan="8" class="text-center py-5">
                                                    <div class="text-muted">
                                                        <i class="fas fa-file-invoice fa-3x mb-3"></i>
                                                        <h5>No invoices found</h5>
                                                        <p>Try adjusting your search criteria or create a new invoice.</p>
                                                        <a href="index.php" class="btn btn-primary">
                                                            <i class="fas fa-plus"></i> Create New Invoice
                                                        </a>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    <!-- Pagination -->
                    <?php if ($total_pages > 1): ?>
                        <nav aria-label="Invoice pagination" class="mt-4">
                            <ul class="pagination">
                                <?php if ($page > 1): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">
                                            <i class="fas fa-chevron-left"></i>
                                        </a>
                                    </li>
                                <?php endif; ?>

                                <?php
                                $start_page = max(1, $page - 2);
                                $end_page = min($total_pages, $page + 2);
                                
                                for ($i = $start_page; $i <= $end_page; $i++):
                                ?>
                                    <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>">
                                            <?= $i ?>
                                        </a>
                                    </li>
                                <?php endfor; ?>

                                <?php if ($page < $total_pages): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">
                                            <i class="fas fa-chevron-right"></i>
                                        </a>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                        
                        <div class="text-center text-muted">
                            Showing <?= ($offset + 1) ?> to <?= min($offset + $records_per_page, $total_records) ?> 
                            of <?= $total_records ?> invoices
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Modals -->
    <!-- Status Update Modal -->
    <div class="modal fade" id="statusModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Update Invoice Status</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Are you sure you want to update the status of this invoice?</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="confirmStatusUpdate">Update Status</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Delete Invoice</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Are you sure you want to delete this invoice? This action cannot be undone.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger" id="confirmDelete">Delete Invoice</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
    <script>
        let selectedInvoices = [];
        let currentInvoiceId = null;
        let currentStatus = null;

        // Selection functions
        function toggleSelectAll() {
            const selectAll = document.getElementById('selectAll');
            const checkboxes = document.querySelectorAll('.invoice-checkbox');
            
            checkboxes.forEach(checkbox => {
                checkbox.checked = selectAll.checked;
            });
            
            updateSelection();
        }

        function updateSelection() {
            const checkboxes = document.querySelectorAll('.invoice-checkbox:checked');
            selectedInvoices = Array.from(checkboxes).map(cb => cb.value);
            
            const bulkActions = document.getElementById('bulkActions');
            const selectedCount = document.getElementById('selectedCount');
            
            if (selectedInvoices.length > 0) {
                bulkActions.style.display = 'block';
                selectedCount.textContent = selectedInvoices.length;
            } else {
                bulkActions.style.display = 'none';
            }

            // Update row highlighting
            document.querySelectorAll('.invoice-row').forEach(row => {
                const checkbox = row.querySelector('.invoice-checkbox');
                if (checkbox.checked) {
                    row.classList.add('selected');
                } else {
                    row.classList.remove('selected');
                }
            });
        }

        function clearSelection() {
            document.querySelectorAll('.invoice-checkbox').forEach(cb => cb.checked = false);
            document.getElementById('selectAll').checked = false;
            updateSelection();
        }

        // Status update functions
        function updateInvoiceStatus(invoiceId, status) {
            currentInvoiceId = invoiceId;
            currentStatus = status;
            
            const modal = new bootstrap.Modal(document.getElementById('statusModal'));
            modal.show();
        }

        function bulkUpdateStatus(status) {
            if (selectedInvoices.length === 0) {
                alert('Please select at least one invoice.');
                return;
            }
            
            if (confirm(`Are you sure you want to mark ${selectedInvoices.length} invoice(s) as ${status}?`)) {
                // Send AJAX request to update status
                fetch('api/update_invoice_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        invoice_ids: selectedInvoices,
                        status: status
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error updating invoices: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while updating invoices.');
                });
            }
        }

        // Delete functions
        function deleteInvoice(invoiceId) {
            currentInvoiceId = invoiceId;
            const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
            modal.show();
        }

        function bulkDelete() {
            if (selectedInvoices.length === 0) {
                alert('Please select at least one invoice.');
                return;
            }
            
            if (confirm(`Are you sure you want to delete ${selectedInvoices.length} invoice(s)? This action cannot be undone.`)) {
                // Send AJAX request to delete invoices
                fetch('api/delete_invoices.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        invoice_ids: selectedInvoices
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error deleting invoices: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while deleting invoices.');
                });
            }
        }

        // Other functions
        function printInvoice(invoiceId) {
            window.open(`view_invoice.php?id=${invoiceId}&print=1`, '_blank');
        }

        function duplicateInvoice(invoiceId) {
            if (confirm('Are you sure you want to duplicate this invoice?')) {
                window.location.href = `duplicate_invoice.php?id=${invoiceId}`;
            }
        }

        function exportInvoices() {
            const params = new URLSearchParams(window.location.search);
            params.set('export', '1');
            window.location.href = 'export_invoices.php?' + params.toString();
        }

        // Modal event handlers
        document.getElementById('confirmStatusUpdate').addEventListener('click', function() {
            if (currentInvoiceId && currentStatus) {
                fetch('api/update_invoice_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        invoice_ids: [currentInvoiceId],
                        status: currentStatus
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error updating invoice: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while updating the invoice.');
                });
            }
            
            bootstrap.Modal.getInstance(document.getElementById('statusModal')).hide();
        });

        document.getElementById('confirmDelete').addEventListener('click', function() {
            if (currentInvoiceId) {
                fetch('api/delete_invoices.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        invoice_ids: [currentInvoiceId]
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error deleting invoice: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while deleting the invoice.');
                });
            }
            
            bootstrap.Modal.getInstance(document.getElementById('deleteModal')).hide();
        });

        // Auto-refresh every 5 minutes
        setInterval(function() {
            // Only refresh if no modals are open
            if (!document.querySelector('.modal.show')) {
                location.reload();
            }
        }, 300000);
    </script>
</body>
</html>
