<?php
session_start();
require_once 'config/database.php';
require_once 'classes/Invoice.php';
require_once 'classes/InvoiceItem.php';

$database = new Database();
$db = $database->getConnection();

$invoice = new Invoice($db);
$invoice_item = new InvoiceItem($db);

// Get invoice ID from URL
$invoice_id = $_GET['id'] ?? 0;
$print_mode = isset($_GET['print']);

if (!$invoice_id) {
    header('Location: invoices.php');
    exit;
}

// Get invoice details
$invoice->id = $invoice_id;
$invoice_data = $invoice->readOne();

if (!$invoice_data) {
    header('Location: invoices.php?error=Invoice not found');
    exit;
}

// Get invoice items
$items_stmt = $invoice_item->readByInvoiceId($invoice_id);

// Number to words function
function numberToWords($number) {
    $ones = array(
        0 => 'ZERO', 1 => 'ONE', 2 => 'TWO', 3 => 'THREE', 4 => 'FOUR', 5 => 'FIVE',
        6 => 'SIX', 7 => 'SEVEN', 8 => 'EIGHT', 9 => 'NINE', 10 => 'TEN',
        11 => 'ELEVEN', 12 => 'TWELVE', 13 => 'THIRTEEN', 14 => 'FOURTEEN', 15 => 'FIFTEEN',
        16 => 'SIXTEEN', 17 => 'SEVENTEEN', 18 => 'EIGHTEEN', 19 => 'NINETEEN'
    );
    
    $tens = array(
        2 => 'TWENTY', 3 => 'THIRTY', 4 => 'FORTY', 5 => 'FIFTY',
        6 => 'SIXTY', 7 => 'SEVENTY', 8 => 'EIGHTY', 9 => 'NINETY'
    );
    
    if ($number < 20) {
        return $ones[$number];
    } elseif ($number < 100) {
        return $tens[intval($number / 10)] . ($number % 10 != 0 ? ' ' . $ones[$number % 10] : '');
    } elseif ($number < 1000) {
        return $ones[intval($number / 100)] . ' HUNDRED' . ($number % 100 != 0 ? ' ' . numberToWords($number % 100) : '');
    } elseif ($number < 100000) {
        return numberToWords(intval($number / 1000)) . ' THOUSAND' . ($number % 1000 != 0 ? ' ' . numberToWords($number % 1000) : '');
    } elseif ($number < 10000000) {
        return numberToWords(intval($number / 100000)) . ' LAKH' . ($number % 100000 != 0 ? ' ' . numberToWords($number % 100000) : '');
    } else {
        return numberToWords(intval($number / 10000000)) . ' CRORE' . ($number % 10000000 != 0 ? ' ' . numberToWords($number % 10000000) : '');
    }
}

$amount_in_words = 'INR ' . numberToWords(intval($invoice_data['total_amount'])) . ' ONLY';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice <?= htmlspecialchars($invoice_data['invoice_number']) ?> - ELECTROFIX SOLUTION</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        @media print {
            .no-print { display: none !important; }
            body { font-size: 12px; }
            .invoice-container { box-shadow: none !important; margin: 0; }
            .btn { display: none; }
        }
        
        .invoice-container {
            max-width: 210mm;
            margin: 0 auto;
            background: white;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            font-family: Arial, sans-serif;
        }
        
        .invoice-header {
            background: #f8f9fa;
            border-bottom: 3px solid #007bff;
            padding: 20px;
        }
        
        .company-name {
            font-size: 24px;
            font-weight: bold;
            color: #007bff;
            margin-bottom: 5px;
        }
        
        .invoice-title {
            background: #007bff;
            color: white;
            text-align: center;
            padding: 10px;
            font-size: 18px;
            font-weight: bold;
            margin: 20px 0;
        }
        
        .invoice-details {
            background: #f8f9fa;
            padding: 15px;
            border: 1px solid #dee2e6;
        }
        
        .address-section {
            border: 1px solid #dee2e6;
            padding: 15px;
            height: 100%;
        }
        
        .items-table {
            border-collapse: collapse;
            width: 100%;
            margin: 20px 0;
        }
        
        .items-table th,
        .items-table td {
            border: 1px solid #dee2e6;
            padding: 8px;
            text-align: left;
        }
        
        .items-table th {
            background: #f8f9fa;
            font-weight: bold;
            text-align: center;
        }
        
        .text-right {
            text-align: right;
        }
        
        .total-section {
            background: #f8f9fa;
            padding: 15px;
            border: 1px solid #dee2e6;
        }
        
        .signature-section {
            margin-top: 30px;
            text-align: right;
        }
        
        .terms-section {
            margin-top: 20px;
            padding: 15px;
            background: #f8f9fa;
            border: 1px solid #dee2e6;
        }

        .status-badge {
            position: absolute;
            top: 20px;
            right: 20px;
            font-size: 1.2rem;
            padding: 10px 20px;
            border-radius: 25px;
        }
    </style>
</head>
<body>
    <div class="container-fluid p-4">
        <?php if (!$print_mode): ?>
        <div class="no-print mb-3">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <a href="invoices.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Back to Invoices
                    </a>
                </div>
                <div>
                    <button onclick="window.print()" class="btn btn-primary me-2">
                        <i class="fas fa-print"></i> Print Invoice
                    </button>
                    <a href="edit_invoice.php?id=<?= $invoice_id ?>" class="btn btn-outline-secondary me-2">
                        <i class="fas fa-edit"></i> Edit
                    </a>
                    <div class="btn-group">
                        <button class="btn btn-outline-success dropdown-toggle" type="button" data-bs-toggle="dropdown">
                            <i class="fas fa-cog"></i> Actions
                        </button>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="#" onclick="updateStatus('sent')">
                                <i class="fas fa-paper-plane"></i> Mark as Sent
                            </a></li>
                            <li><a class="dropdown-item" href="#" onclick="updateStatus('paid')">
                                <i class="fas fa-check"></i> Mark as Paid
                            </a></li>
                            <li><a class="dropdown-item" href="duplicate_invoice.php?id=<?= $invoice_id ?>">
                                <i class="fas fa-copy"></i> Duplicate
                            </a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item text-danger" href="#" onclick="deleteInvoice()">
                                <i class="fas fa-trash"></i> Delete
                            </a></li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <div class="invoice-container position-relative">
            <!-- Status Badge -->
            <?php
            $status_colors = [
                'draft' => 'secondary',
                'sent' => 'warning',
                'paid' => 'success',
                'cancelled' => 'danger'
            ];
            $status_color = $status_colors[$invoice_data['status']] ?? 'secondary';
            ?>
            <span class="badge bg-<?= $status_color ?> status-badge no-print">
                <?= ucfirst($invoice_data['status']) ?>
            </span>

            <!-- Header -->
            <div class="invoice-header">
                <div class="row">
                    <div class="col-8">
                        <div class="company-name"><?= htmlspecialchars($invoice_data['company_name']) ?></div>
                        <div><?= htmlspecialchars($invoice_data['company_address']) ?></div>
                        <div>CONTACT: <?= htmlspecialchars($invoice_data['company_contact']) ?></div>
                        <div>Email: <?= htmlspecialchars($invoice_data['company_email']) ?></div>
                    </div>
                    <div class="col-4 text-end">
                        <!-- Logo space -->
                        <div style="width: 100px; height: 80px; border: 2px dashed #ccc; display: inline-block; text-align: center; line-height: 80px; color: #999;">
                            LOGO
                        </div>
                    </div>
                </div>
            </div>

            <!-- Invoice Title -->
            <div class="invoice-title">TAX INVOICE</div>

            <!-- Invoice Details -->
            <div class="row mb-3">
                <div class="col-6">
                    <div class="invoice-details">
                        <div><strong>INVOICE NO:</strong> <?= htmlspecialchars($invoice_data['invoice_number']) ?></div>
                        <div><strong>INVOICE DATE:</strong> <?= date('d/m/Y', strtotime($invoice_data['invoice_date'])) ?></div>
                        <div><strong>RGP NO:</strong> <?= htmlspecialchars($invoice_data['rgp_number']) ?></div>
                        <div><strong>RGP DATE:</strong> <?= $invoice_data['rgp_date'] ? date('d/m/Y', strtotime($invoice_data['rgp_date'])) : '-' ?></div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="invoice-details">
                        <div><strong>PO NO:</strong> <?= htmlspecialchars($invoice_data['po_number']) ?></div>
                        <div><strong>PO DATE:</strong> <?= $invoice_data['po_date'] ? date('d/m/Y', strtotime($invoice_data['po_date'])) : '-' ?></div>
                        <div><strong>GST NO:</strong> <?= htmlspecialchars($invoice_data['company_gst']) ?></div>
                        <div><strong>STATE:</strong> <?= htmlspecialchars($invoice_data['company_state']) ?></div>
                    </div>
                </div>
            </div>

            <div class="mb-3">
                <strong>Tax is payable on reverse charge:</strong> <?= htmlspecialchars($invoice_data['reverse_charge']) ?>
            </div>

            <!-- Addresses -->
            <div class="row mb-3">
                <div class="col-6">
                    <div class="address-section">
                        <h6><strong>Billed to:</strong></h6>
                        <div><strong>Name:</strong> <?= htmlspecialchars($invoice_data['customer_name']) ?></div>
                        <div><strong>Address:</strong> <?= htmlspecialchars($invoice_data['billing_address']) ?></div>
                        <div><strong>GSTIN:</strong> <?= htmlspecialchars($invoice_data['customer_gstin']) ?></div>
                        <div><strong>STATE:</strong> <?= htmlspecialchars($invoice_data['customer_state']) ?></div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="address-section">
                        <h6><strong>Shipped to:</strong></h6>
                        <div><strong>Name:</strong> <?= htmlspecialchars($invoice_data['customer_name']) ?></div>
                        <div><strong>Address:</strong> <?= htmlspecialchars($invoice_data['shipping_address']) ?></div>
                        <div><strong>GSTIN:</strong> <?= htmlspecialchars($invoice_data['customer_gstin']) ?></div>
                        <div><strong>STATE:</strong> <?= htmlspecialchars($invoice_data['customer_state']) ?></div>
                    </div>
                </div>
            </div>

            <!-- Items Table -->
            <table class="items-table">
                <thead>
                    <tr>
                        <th>S.NO.</th>
                        <th>Product Description/Service</th>
                        <th>HSN/SAC</th>
                        <th>QTY.</th>
                        <th>RATE/PCS</th>
                        <th>Total Amount</th>
                    </tr>
                </thead>
                <tbody>
                    <?php $sno = 1; ?>
                    <?php while ($item = $items_stmt->fetch(PDO::FETCH_ASSOC)): ?>
                        <tr>
                            <td class="text-center"><?= $sno++ ?></td>
                            <td><?= htmlspecialchars($item['description']) ?></td>
                            <td class="text-center"><?= htmlspecialchars($item['hsn_sac']) ?></td>
                            <td class="text-center"><?= number_format($item['quantity'], 0) ?></td>
                            <td class="text-right"><?= number_format($item['rate'], 2) ?></td>
                            <td class="text-right"><?= number_format($item['amount'], 2) ?></td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>

            <!-- Totals Section -->
            <div class="row">
                <div class="col-6">
                    <div class="total-section">
                        <h6><strong>Bank Details:</strong></h6>
                        <div><strong>Bank Name:</strong> <?= htmlspecialchars($invoice_data['bank_name']) ?></div>
                        <div><strong>Bank Account Number:</strong> <?= htmlspecialchars($invoice_data['bank_account']) ?></div>
                        <div><strong>Bank IFSC:</strong> <?= htmlspecialchars($invoice_data['bank_ifsc']) ?></div>
                        
                        <div class="mt-3">
                            <strong>Total invoice amount in words:</strong><br>
                            <?= $amount_in_words ?>
                        </div>
                    </div>
                </div>
                <div class="col-6">
                    <div class="total-section">
                        <div class="d-flex justify-content-between">
                            <span><strong>Total amount before tax:</strong></span>
                            <span><?= number_format($invoice_data['subtotal'], 2) ?></span>
                        </div>
                        
                        <?php if ($invoice_data['tax_type'] === 'cgst_sgst'): ?>
                            <div class="d-flex justify-content-between">
                                <span>Add: CGST @9%</span>
                                <span><?= number_format($invoice_data['cgst_amount'], 2) ?></span>
                            </div>
                            <div class="d-flex justify-content-between">
                                <span>Add: SGST @9%</span>
                                <span><?= number_format($invoice_data['sgst_amount'], 2) ?></span>
                            </div>
                        <?php else: ?>
                            <div class="d-flex justify-content-between">
                                <span>Add: IGST @18%</span>
                                <span><?= number_format($invoice_data['igst_amount'], 2) ?></span>
                            </div>
                        <?php endif; ?>
                        
                        <hr>
                        <div class="d-flex justify-content-between">
                            <span><strong>Total tax amount:</strong></span>
                            <span><strong><?= number_format($invoice_data['total_tax'], 2) ?></strong></span>
                        </div>
                        <div class="d-flex justify-content-between">
                            <span><strong>Total amount after tax:</strong></span>
                            <span><strong><?= number_format($invoice_data['total_amount'], 2) ?></strong></span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Terms and Signature -->
            <div class="terms-section">
                <div class="row">
                    <div class="col-6">
                        <h6><strong>PAYMENT TERMS & CONDITIONS</strong></h6>
                        <p><?= htmlspecialchars($invoice_data['terms_conditions'] ?? 'Net 10 - Payment ten days after the invoice date') ?></p>
                        <p><strong>All Disputes Are Subject to Noida Jurisdiction</strong></p>
                    </div>
                    <div class="col-6">
                        <div class="signature-section">
                            <p><strong>For: <?= htmlspecialchars($invoice_data['company_name']) ?></strong></p>
                            <div style="height: 60px;"></div>
                            <p><strong>Authorized Signatory</strong></p>
                            <p><strong>E. & O.E.</strong></p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function updateStatus(status) {
            if (confirm(`Are you sure you want to mark this invoice as ${status}?`)) {
                fetch('api/update_invoice_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        invoice_ids: [<?= $invoice_id ?>],
                        status: status
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error updating invoice: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while updating the invoice.');
                });
            }
        }

        function deleteInvoice() {
            if (confirm('Are you sure you want to delete this invoice? This action cannot be undone.')) {
                fetch('api/delete_invoices.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        invoice_ids: [<?= $invoice_id ?>]
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        window.location.href = 'invoices.php';
                    } else {
                        alert('Error deleting invoice: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while deleting the invoice.');
                });
            }
        }

        <?php if ($print_mode): ?>
        // Auto-print when in print mode
        window.onload = function() {
            window.print();
        }
        <?php endif; ?>
    </script>
</body>
</html>
