<?php
session_start();
require_once 'config/database.php';
require_once 'classes/Customer.php';

$database = new Database();
$db = $database->getConnection();

$customer = new Customer($db);

// Handle form submission
if ($_POST) {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                $customer->name = $_POST['name'];
                $customer->billing_address = $_POST['billing_address'];
                $customer->shipping_address = $_POST['shipping_address'];
                $customer->gstin = $_POST['gstin'];
                $customer->state = $_POST['state'];
                $customer->contact = $_POST['contact'];
                $customer->email = $_POST['email'];
                
                if ($customer->create()) {
                    $success_message = "Customer created successfully!";
                } else {
                    $error_message = "Failed to create customer.";
                }
                break;
                
            case 'update':
                $customer->id = $_POST['customer_id'];
                $customer->name = $_POST['name'];
                $customer->billing_address = $_POST['billing_address'];
                $customer->shipping_address = $_POST['shipping_address'];
                $customer->gstin = $_POST['gstin'];
                $customer->state = $_POST['state'];
                $customer->contact = $_POST['contact'];
                $customer->email = $_POST['email'];
                
                if ($customer->update()) {
                    $success_message = "Customer updated successfully!";
                } else {
                    $error_message = "Failed to update customer.";
                }
                break;
                
            case 'delete':
                $customer->id = $_POST['customer_id'];
                if ($customer->delete()) {
                    $success_message = "Customer deleted successfully!";
                } else {
                    $error_message = "Failed to delete customer.";
                }
                break;
        }
    }
}

// Handle search
$search = $_GET['search'] ?? '';
if (!empty($search)) {
    $customers_stmt = $customer->search($search);
} else {
    $customers_stmt = $customer->readAll();
}

// Get customer for editing
$edit_customer = null;
if (isset($_GET['edit'])) {
    $edit_customer = new Customer($db);
    $edit_customer->id = $_GET['edit'];
    $edit_customer->readOne();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Customer Management - ELECTROFIX SOLUTION</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .sidebar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .nav-link {
            color: rgba(255, 255, 255, 0.8);
            transition: all 0.3s ease;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 10px;
        }
        .customer-card {
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            transition: transform 0.3s ease;
        }
        .customer-card:hover {
            transform: translateY(-2px);
        }
        .form-container {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar p-0">
                <div class="p-4">
                    <h4 class="text-white mb-4">
                        <i class="fas fa-bolt"></i> ELECTROFIX
                    </h4>
                    <nav class="nav flex-column">
                        <a class="nav-link mb-2" href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-2"></i> Dashboard
                        </a>
                        <a class="nav-link mb-2" href="index.php">
                            <i class="fas fa-plus me-2"></i> New Invoice
                        </a>
                        <a class="nav-link mb-2" href="invoices.php">
                            <i class="fas fa-file-invoice me-2"></i> All Invoices
                        </a>
                        <a class="nav-link active mb-2" href="customers.php">
                            <i class="fas fa-users me-2"></i> Customers
                        </a>
                        <a class="nav-link mb-2" href="products.php">
                            <i class="fas fa-box me-2"></i> Products
                        </a>
                        <a class="nav-link mb-2" href="reports.php">
                            <i class="fas fa-chart-bar me-2"></i> Reports
                        </a>
                        <a class="nav-link mb-2" href="settings.php">
                            <i class="fas fa-cog me-2"></i> Settings
                        </a>
                    </nav>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 col-lg-10">
                <div class="p-4">
                    <!-- Header -->
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <h2>Customer Management</h2>
                            <p class="text-muted">Manage your customer database</p>
                        </div>
                        <div>
                            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#customerModal">
                                <i class="fas fa-plus"></i> Add Customer
                            </button>
                        </div>
                    </div>

                    <!-- Alerts -->
                    <?php if (isset($success_message)): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <?= $success_message ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <?php if (isset($error_message)): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <?= $error_message ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <!-- Search -->
                    <div class="form-container">
                        <form method="GET" class="row g-3">
                            <div class="col-md-8">
                                <input type="text" class="form-control" name="search" 
                                       placeholder="Search customers by name, GSTIN, or contact..." 
                                       value="<?= htmlspecialchars($search) ?>">
                            </div>
                            <div class="col-md-2">
                                <button type="submit" class="btn btn-primary w-100">
                                    <i class="fas fa-search"></i> Search
                                </button>
                            </div>
                            <div class="col-md-2">
                                <a href="customers.php" class="btn btn-outline-secondary w-100">
                                    <i class="fas fa-times"></i> Clear
                                </a>
                            </div>
                        </form>
                    </div>

                    <!-- Customers Table -->
                    <div class="card customer-card">
                        <div class="card-body p-0">
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Customer Name</th>
                                            <th>GSTIN</th>
                                            <th>Contact</th>
                                            <th>Email</th>
                                            <th>State</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if ($customers_stmt->rowCount() > 0): ?>
                                            <?php while ($row = $customers_stmt->fetch(PDO::FETCH_ASSOC)): ?>
                                                <tr>
                                                    <td>
                                                        <strong><?= htmlspecialchars($row['name']) ?></strong>
                                                        <br>
                                                        <small class="text-muted">
                                                            <?= htmlspecialchars(substr($row['billing_address'], 0, 50)) ?>...
                                                        </small>
                                                    </td>
                                                    <td><?= htmlspecialchars($row['gstin']) ?></td>
                                                    <td><?= htmlspecialchars($row['contact']) ?></td>
                                                    <td><?= htmlspecialchars($row['email']) ?></td>
                                                    <td><?= htmlspecialchars($row['state']) ?></td>
                                                    <td>
                                                        <button class="btn btn-sm btn-outline-primary" 
                                                                onclick="viewCustomer(<?= $row['id'] ?>)" title="View">
                                                            <i class="fas fa-eye"></i>
                                                        </button>
                                                        <button class="btn btn-sm btn-outline-secondary" 
                                                                onclick="editCustomer(<?= $row['id'] ?>)" title="Edit">
                                                            <i class="fas fa-edit"></i>
                                                        </button>
                                                        <button class="btn btn-sm btn-outline-danger" 
                                                                onclick="deleteCustomer(<?= $row['id'] ?>)" title="Delete">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                            <?php endwhile; ?>
                                        <?php else: ?>
                                            <tr>
                                                <td colspan="6" class="text-center py-5">
                                                    <div class="text-muted">
                                                        <i class="fas fa-users fa-3x mb-3"></i>
                                                        <h5>No customers found</h5>
                                                        <p>Add your first customer to get started.</p>
                                                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#customerModal">
                                                            <i class="fas fa-plus"></i> Add Customer
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Customer Modal -->
    <div class="modal fade" id="customerModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="customerModalTitle">Add New Customer</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" id="customerForm">
                    <div class="modal-body">
                        <input type="hidden" name="action" id="formAction" value="create">
                        <input type="hidden" name="customer_id" id="customerId">
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Customer Name *</label>
                                <input type="text" class="form-control" name="name" id="customerName" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">GSTIN</label>
                                <input type="text" class="form-control" name="gstin" id="customerGstin">
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Contact</label>
                                <input type="text" class="form-control" name="contact" id="customerContact">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Email</label>
                                <input type="email" class="form-control" name="email" id="customerEmail">
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">State</label>
                            <input type="text" class="form-control" name="state" id="customerState">
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Billing Address *</label>
                            <textarea class="form-control" name="billing_address" id="billingAddress" rows="3" required></textarea>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Shipping Address</label>
                            <textarea class="form-control" name="shipping_address" id="shippingAddress" rows="3"></textarea>
                            <div class="form-check mt-2">
                                <input class="form-check-input" type="checkbox" id="sameAsBilling">
                                <label class="form-check-label" for="sameAsBilling">
                                    Same as billing address
                                </label>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary" id="submitBtn">Save Customer</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- View Customer Modal -->
    <div class="modal fade" id="viewCustomerModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Customer Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="customerDetails">
                    <!-- Customer details will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Same as billing address checkbox
        document.getElementById('sameAsBilling').addEventListener('change', function() {
            if (this.checked) {
                document.getElementById('shippingAddress').value = document.getElementById('billingAddress').value;
            }
        });

        // Update shipping address when billing address changes
        document.getElementById('billingAddress').addEventListener('input', function() {
            if (document.getElementById('sameAsBilling').checked) {
                document.getElementById('shippingAddress').value = this.value;
            }
        });

        function editCustomer(customerId) {
            // Fetch customer data and populate form
            fetch(`api/get_customer.php?id=${customerId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('customerModalTitle').textContent = 'Edit Customer';
                        document.getElementById('formAction').value = 'update';
                        document.getElementById('customerId').value = data.customer.id;
                        document.getElementById('customerName').value = data.customer.name;
                        document.getElementById('customerGstin').value = data.customer.gstin;
                        document.getElementById('customerContact').value = data.customer.contact;
                        document.getElementById('customerEmail').value = data.customer.email;
                        document.getElementById('customerState').value = data.customer.state;
                        document.getElementById('billingAddress').value = data.customer.billing_address;
                        document.getElementById('shippingAddress').value = data.customer.shipping_address;
                        document.getElementById('submitBtn').textContent = 'Update Customer';
                        
                        new bootstrap.Modal(document.getElementById('customerModal')).show();
                    }
                })
                .catch(error => console.error('Error:', error));
        }

        function viewCustomer(customerId) {
            // Fetch customer data and display
            fetch(`api/get_customer.php?id=${customerId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const customer = data.customer;
                        document.getElementById('customerDetails').innerHTML = `
                            <div class="row">
                                <div class="col-md-6">
                                    <h6>Basic Information</h6>
                                    <p><strong>Name:</strong> ${customer.name}</p>
                                    <p><strong>GSTIN:</strong> ${customer.gstin || 'N/A'}</p>
                                    <p><strong>Contact:</strong> ${customer.contact || 'N/A'}</p>
                                    <p><strong>Email:</strong> ${customer.email || 'N/A'}</p>
                                    <p><strong>State:</strong> ${customer.state || 'N/A'}</p>
                                </div>
                                <div class="col-md-6">
                                    <h6>Addresses</h6>
                                    <p><strong>Billing Address:</strong><br>${customer.billing_address}</p>
                                    <p><strong>Shipping Address:</strong><br>${customer.shipping_address || 'Same as billing'}</p>
                                </div>
                            </div>
                        `;
                        new bootstrap.Modal(document.getElementById('viewCustomerModal')).show();
                    }
                })
                .catch(error => console.error('Error:', error));
        }

        function deleteCustomer(customerId) {
            if (confirm('Are you sure you want to delete this customer? This action cannot be undone.')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="customer_id" value="${customerId}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }

        // Reset form when modal is closed
        document.getElementById('customerModal').addEventListener('hidden.bs.modal', function() {
            document.getElementById('customerForm').reset();
            document.getElementById('customerModalTitle').textContent = 'Add New Customer';
            document.getElementById('formAction').value = 'create';
            document.getElementById('customerId').value = '';
            document.getElementById('submitBtn').textContent = 'Save Customer';
        });
    </script>
</body>
</html>
