<?php
session_start();
require_once 'config/database.php';
require_once 'classes/Invoice.php';
require_once 'classes/Customer.php';
require_once 'classes/InvoiceItem.php';

// Sample data for demonstration
$invoice_data = [
    'company' => [
        'name' => 'ELECTROFIX SOLUTION',
        'address' => 'D-171, 2nd Floor, Sector-10, Noida-201301',
        'contact' => '8851369694',
        'email' => 'contact.electrofixsolution@gmail.com',
        'gst' => '09DMFPP9986Q1ZJ',
        'state' => 'UTTAR PRADESH 09'
    ],
    'bank' => [
        'name' => 'HDFC Bank Ltd.',
        'account' => '50200086529101',
        'ifsc' => 'HDFC0000088'
    ]
];

// Handle form submission
if ($_POST) {
    try {
        $database = new Database();
        $db = $database->getConnection();
        
        $invoice = new Invoice($db);
        $customer = new Customer($db);
        $invoice_item = new InvoiceItem($db);
        
        $db->beginTransaction();
        
        // Check if customer exists or create new
        $customer_stmt = $customer->search($_POST['customer_name']);
        $existing_customer = $customer_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing_customer) {
            $customer_id = $existing_customer['id'];
        } else {
            // Create new customer
            $customer->name = $_POST['customer_name'];
            $customer->billing_address = $_POST['billing_address'];
            $customer->shipping_address = $_POST['shipping_address'];
            $customer->gstin = $_POST['customer_gstin'];
            $customer->state = $_POST['customer_state'];
            $customer->contact = '';
            $customer->email = '';
            
            $customer_id = $customer->create();
        }
        
        // Calculate totals
        $subtotal = 0;
        if (isset($_POST['items'])) {
            foreach ($_POST['items'] as $item) {
                $subtotal += ($item['qty'] * $item['rate']);
            }
        }
        
        $tax_rate = 0.18;
        $tax_amount = $subtotal * $tax_rate;
        $total_amount = $subtotal + $tax_amount;
        
        // Create invoice
        $invoice->invoice_number = $_POST['invoice_no'];
        $invoice->invoice_date = $_POST['invoice_date'];
        $invoice->due_date = null;
        $invoice->company_id = 1; // Default company
        $invoice->customer_id = $customer_id;
        $invoice->rgp_number = $_POST['rgp_no'];
        $invoice->rgp_date = $_POST['rgp_date'] ? $_POST['rgp_date'] : null;
        $invoice->po_number = $_POST['po_no'];
        $invoice->po_date = $_POST['po_date'] ? $_POST['po_date'] : null;
        $invoice->reverse_charge = $_POST['reverse_charge'];
        $invoice->tax_type = $_POST['tax_type'];
        $invoice->subtotal = $subtotal;
        
        if ($_POST['tax_type'] == 'cgst_sgst') {
            $invoice->cgst_amount = $tax_amount / 2;
            $invoice->sgst_amount = $tax_amount / 2;
            $invoice->igst_amount = 0;
        } else {
            $invoice->cgst_amount = 0;
            $invoice->sgst_amount = 0;
            $invoice->igst_amount = $tax_amount;
        }
        
        $invoice->total_tax = $tax_amount;
        $invoice->total_amount = $total_amount;
        $invoice->amount_in_words = numberToWords($total_amount);
        $invoice->notes = '';
        $invoice->terms_conditions = 'Net 10 - Payment ten days after the invoice date';
        $invoice->status = 'draft';
        
        $invoice_id = $invoice->create();
        
        if ($invoice_id && isset($_POST['items'])) {
            // Create invoice items
            foreach ($_POST['items'] as $item) {
                $invoice_item->invoice_id = $invoice_id;
                $invoice_item->product_id = null;
                $invoice_item->description = $item['description'];
                $invoice_item->hsn_sac = $item['hsn'];
                $invoice_item->quantity = $item['qty'];
                $invoice_item->unit = 'PCS';
                $invoice_item->rate = $item['rate'];
                $invoice_item->amount = $item['qty'] * $item['rate'];
                
                $invoice_item->create();
            }
        }
        
        $db->commit();
        
        // Store invoice data in session for display
        $_SESSION['invoice_data'] = $_POST;
        $_SESSION['invoice_id'] = $invoice_id;
        
        // Redirect to invoice display
        header('Location: invoice.php');
        exit;
        
    } catch (Exception $e) {
        if (isset($db)) {
            $db->rollback();
        }
        $error_message = "Error creating invoice: " . $e->getMessage();
    }
}

// Number to words function
function numberToWords($number) {
    $ones = array(
        0 => 'ZERO', 1 => 'ONE', 2 => 'TWO', 3 => 'THREE', 4 => 'FOUR', 5 => 'FIVE',
        6 => 'SIX', 7 => 'SEVEN', 8 => 'EIGHT', 9 => 'NINE', 10 => 'TEN',
        11 => 'ELEVEN', 12 => 'TWELVE', 13 => 'THIRTEEN', 14 => 'FOURTEEN', 15 => 'FIFTEEN',
        16 => 'SIXTEEN', 17 => 'SEVENTEEN', 18 => 'EIGHTEEN', 19 => 'NINETEEN'
    );
    
    $tens = array(
        2 => 'TWENTY', 3 => 'THIRTY', 4 => 'FORTY', 5 => 'FIFTY',
        6 => 'SIXTY', 7 => 'SEVENTY', 8 => 'EIGHTY', 9 => 'NINETY'
    );
    
    if ($number < 20) {
        return $ones[$number];
    } elseif ($number < 100) {
        return $tens[intval($number / 10)] . ($number % 10 != 0 ? ' ' . $ones[$number % 10] : '');
    } elseif ($number < 1000) {
        return $ones[intval($number / 100)] . ' HUNDRED' . ($number % 100 != 0 ? ' ' . numberToWords($number % 100) : '');
    } elseif ($number < 100000) {
        return numberToWords(intval($number / 1000)) . ' THOUSAND' . ($number % 1000 != 0 ? ' ' . numberToWords($number % 1000) : '');
    } elseif ($number < 10000000) {
        return numberToWords(intval($number / 100000)) . ' LAKH' . ($number % 100000 != 0 ? ' ' . numberToWords($number % 100000) : '');
    } else {
        return numberToWords(intval($number / 10000000)) . ' CRORE' . ($number % 10000000 != 0 ? ' ' . numberToWords($number % 10000000) : '');
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice Generator - ELECTROFIX SOLUTION</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .form-container {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 30px;
            margin: 20px 0;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
        }
        .section-header {
            background: #007bff;
            color: white;
            padding: 10px 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .btn-add-item {
            background: #28a745;
            border: none;
            border-radius: 50px;
            padding: 10px 20px;
        }
        .item-row {
            background: white;
            border: 1px solid #dee2e6;
            border-radius: 5px;
            padding: 15px;
            margin-bottom: 10px;
        }
        .remove-item {
            color: #dc3545;
            cursor: pointer;
            font-size: 18px;
        }
        .sidebar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .nav-link {
            color: rgba(255, 255, 255, 0.8);
            transition: all 0.3s ease;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar p-0">
                <div class="p-4">
                    <h4 class="text-white mb-4">
                        <i class="fas fa-bolt"></i> ELECTROFIX
                    </h4>
                    <nav class="nav flex-column">
                        <a class="nav-link mb-2" href="dashboard.php">
                            <i class="fas fa-tachometer-alt me-2"></i> Dashboard
                        </a>
                        <a class="nav-link active mb-2" href="index.php">
                            <i class="fas fa-plus me-2"></i> New Invoice
                        </a>
                        <a class="nav-link mb-2" href="invoices.php">
                            <i class="fas fa-file-invoice me-2"></i> All Invoices
                        </a>
                        <a class="nav-link mb-2" href="customers.php">
                            <i class="fas fa-users me-2"></i> Customers
                        </a>
                        <a class="nav-link mb-2" href="products.php">
                            <i class="fas fa-box me-2"></i> Products
                        </a>
                        <a class="nav-link mb-2" href="reports.php">
                            <i class="fas fa-chart-bar me-2"></i> Reports
                        </a>
                        <a class="nav-link mb-2" href="settings.php">
                            <i class="fas fa-cog me-2"></i> Settings
                        </a>
                    </nav>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 col-lg-10">
                <!-- Add this after the main content div starts -->
                <?php if (isset($error_message)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?= $error_message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                <div class="p-4">
                    <form method="POST" id="invoiceForm">
                        <div class="row">
                            <div class="col-lg-6">
                                <div class="form-container">
                                    <div class="section-header">
                                        <h4><i class="fas fa-file-invoice"></i> Invoice Details</h4>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">Invoice No</label>
                                            <input type="text" class="form-control" name="invoice_no" value="002/2025-26" required>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">Invoice Date</label>
                                            <input type="date" class="form-control" name="invoice_date" value="<?= date('Y-m-d') ?>" required>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">RGP No</label>
                                            <input type="text" class="form-control" name="rgp_no" value="MDC072400412">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">RGP Date</label>
                                            <input type="date" class="form-control" name="rgp_date" value="2025-02-15">
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">PO No</label>
                                            <input type="text" class="form-control" name="po_no" value="4400005720">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">PO Date</label>
                                            <input type="date" class="form-control" name="po_date" value="2025-03-19">
                                        </div>
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Tax on Reverse Charge</label>
                                        <select class="form-control" name="reverse_charge">
                                            <option value="NO">NO</option>
                                            <option value="YES">YES</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <div class="col-lg-6">
                                <div class="form-container">
                                    <div class="section-header">
                                        <h4><i class="fas fa-user"></i> Customer Details</h4>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Customer Name</label>
                                        <input type="text" class="form-control" name="customer_name" value="BHAGWATI PRODUCTS LIMITED" required>
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Billing Address</label>
                                        <textarea class="form-control" name="billing_address" rows="3" required>Entire tower Tec-1,Entire tower Tec-2, Entire Tower WTC one C, TZ-13A, IT park, sector-techzone, Greater Noida, Gautambuddha Nagar, Uttar Pradesh, 201308</textarea>
                                    </div>

                                    <div class="mb-3">
                                        <label class="form-label">Shipping Address</label>
                                        <textarea class="form-control" name="shipping_address" rows="3" required>Entire tower Tec-1,Entire tower Tec-2, Entire Tower WTC one C, TZ-13A, IT park, sector-techzone, Greater Noida, Gautambuddha Nagar, Uttar Pradesh, 201308</textarea>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">Customer GSTIN</label>
                                            <input type="text" class="form-control" name="customer_gstin" value="09AACCB1828N1ZG" required>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">Customer State</label>
                                            <input type="text" class="form-control" name="customer_state" value="NOIDA 09" required>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-12">
                                <div class="form-container">
                                    <div class="section-header d-flex justify-content-between align-items-center">
                                        <h4><i class="fas fa-shopping-cart"></i> Invoice Items</h4>
                                        <button type="button" class="btn btn-add-item text-white" onclick="addItem()">
                                            <i class="fas fa-plus"></i> Add Item
                                        </button>
                                    </div>
                                    
                                    <div id="items-container">
                                        <!-- Items will be added here -->
                                    </div>

                                    <div class="row mt-4">
                                        <div class="col-md-4">
                                            <label class="form-label">Tax Type</label>
                                            <select class="form-control" name="tax_type" id="tax_type" onchange="calculateTotals()">
                                                <option value="cgst_sgst">CGST + SGST (9% + 9%)</option>
                                                <option value="igst">IGST (18%)</option>
                                            </select>
                                        </div>
                                        <div class="col-md-8">
                                            <div class="row">
                                                <div class="col-md-4">
                                                    <label class="form-label">Subtotal</label>
                                                    <input type="text" class="form-control" id="subtotal" readonly>
                                                </div>
                                                <div class="col-md-4">
                                                    <label class="form-label">Tax Amount</label>
                                                    <input type="text" class="form-control" id="tax_amount" readonly>
                                                </div>
                                                <div class="col-md-4">
                                                    <label class="form-label">Total Amount</label>
                                                    <input type="text" class="form-control" id="total_amount" readonly>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-12 text-center">
                                <button type="submit" class="btn btn-primary btn-lg px-5">
                                    <i class="fas fa-file-invoice"></i> Generate Invoice
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let itemCount = 0;

        function addItem() {
            itemCount++;
            const container = document.getElementById('items-container');
            const itemHtml = `
                <div class="item-row" id="item-${itemCount}">
                    <div class="row">
                        <div class="col-md-4 mb-2">
                            <label class="form-label">Product Description</label>
                            <input type="text" class="form-control" name="items[${itemCount}][description]" required>
                        </div>
                        <div class="col-md-2 mb-2">
                            <label class="form-label">HSN/SAC</label>
                            <input type="text" class="form-control" name="items[${itemCount}][hsn]">
                        </div>
                        <div class="col-md-2 mb-2">
                            <label class="form-label">Quantity</label>
                            <input type="number" class="form-control item-qty" name="items[${itemCount}][qty]" onchange="calculateItemTotal(${itemCount})" required>
                        </div>
                        <div class="col-md-2 mb-2">
                            <label class="form-label">Rate</label>
                            <input type="number" step="0.01" class="form-control item-rate" name="items[${itemCount}][rate]" onchange="calculateItemTotal(${itemCount})" required>
                        </div>
                        <div class="col-md-1 mb-2">
                            <label class="form-label">Total</label>
                            <input type="text" class="form-control item-total" id="total-${itemCount}" readonly>
                        </div>
                        <div class="col-md-1 mb-2 d-flex align-items-end">
                            <span class="remove-item" onclick="removeItem(${itemCount})" title="Remove Item">
                                <i class="fas fa-trash"></i>
                            </span>
                        </div>
                    </div>
                </div>
            `;
            container.insertAdjacentHTML('beforeend', itemHtml);
        }

        function removeItem(itemId) {
            document.getElementById(`item-${itemId}`).remove();
            calculateTotals();
        }

        function calculateItemTotal(itemId) {
            const qtyInput = document.querySelector(`input[name="items[${itemId}][qty]"]`);
            const rateInput = document.querySelector(`input[name="items[${itemId}][rate]"]`);
            const totalInput = document.getElementById(`total-${itemId}`);
            
            const qty = parseFloat(qtyInput.value) || 0;
            const rate = parseFloat(rateInput.value) || 0;
            const total = qty * rate;
            
            totalInput.value = total.toFixed(2);
            calculateTotals();
        }

        function calculateTotals() {
            let subtotal = 0;
            const totalInputs = document.querySelectorAll('.item-total');
            
            totalInputs.forEach(input => {
                subtotal += parseFloat(input.value) || 0;
            });
            
            const taxType = document.getElementById('tax_type').value;
            let taxAmount = 0;
            
            if (taxType === 'cgst_sgst') {
                taxAmount = subtotal * 0.18; // 9% + 9%
            } else {
                taxAmount = subtotal * 0.18; // 18%
            }
            
            const totalAmount = subtotal + taxAmount;
            
            document.getElementById('subtotal').value = subtotal.toFixed(2);
            document.getElementById('tax_amount').value = taxAmount.toFixed(2);
            document.getElementById('total_amount').value = totalAmount.toFixed(2);
        }

        // Add initial item
        addItem();
    </script>
</body>
</html>
