-- Create database if it doesn't exist
CREATE DATABASE IF NOT EXISTS invoice_system;
USE invoice_system;

-- Companies table (for multi-company support)
CREATE TABLE IF NOT EXISTS companies (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255) NOT NULL,
    address TEXT,
    contact VARCHAR(20),
    email VARCHAR(100),
    gst_number VARCHAR(20),
    state VARCHAR(100),
    bank_name VARCHAR(255),
    bank_account VARCHAR(50),
    bank_ifsc VARCHAR(20),
    logo_path VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Customers table
CREATE TABLE IF NOT EXISTS customers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255) NOT NULL,
    billing_address TEXT,
    shipping_address TEXT,
    gstin VARCHAR(20),
    state VARCHAR(100),
    contact VARCHAR(20),
    email VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Products/Services table
CREATE TABLE IF NOT EXISTS products (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    hsn_sac VARCHAR(20),
    unit VARCHAR(20) DEFAULT 'PCS',
    rate DECIMAL(10,2) DEFAULT 0.00,
    tax_rate DECIMAL(5,2) DEFAULT 18.00,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Invoices table
CREATE TABLE IF NOT EXISTS invoices (
    id INT PRIMARY KEY AUTO_INCREMENT,
    invoice_number VARCHAR(50) UNIQUE NOT NULL,
    invoice_date DATE NOT NULL,
    due_date DATE,
    company_id INT DEFAULT 1,
    customer_id INT NOT NULL,
    rgp_number VARCHAR(50),
    rgp_date DATE,
    po_number VARCHAR(50),
    po_date DATE,
    reverse_charge ENUM('YES', 'NO') DEFAULT 'NO',
    tax_type ENUM('cgst_sgst', 'igst') DEFAULT 'cgst_sgst',
    subtotal DECIMAL(12,2) DEFAULT 0.00,
    cgst_amount DECIMAL(12,2) DEFAULT 0.00,
    sgst_amount DECIMAL(12,2) DEFAULT 0.00,
    igst_amount DECIMAL(12,2) DEFAULT 0.00,
    total_tax DECIMAL(12,2) DEFAULT 0.00,
    total_amount DECIMAL(12,2) DEFAULT 0.00,
    amount_in_words TEXT,
    notes TEXT,
    terms_conditions TEXT,
    status ENUM('draft', 'sent', 'paid', 'cancelled') DEFAULT 'draft',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (customer_id) REFERENCES customers(id)
);

-- Invoice items table
CREATE TABLE IF NOT EXISTS invoice_items (
    id INT PRIMARY KEY AUTO_INCREMENT,
    invoice_id INT NOT NULL,
    product_id INT,
    description VARCHAR(500) NOT NULL,
    hsn_sac VARCHAR(20),
    quantity DECIMAL(10,2) NOT NULL,
    unit VARCHAR(20) DEFAULT 'PCS',
    rate DECIMAL(10,2) NOT NULL,
    amount DECIMAL(12,2) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (invoice_id) REFERENCES invoices(id) ON DELETE CASCADE
);

-- Invoice sequence table (for auto-generating invoice numbers)
CREATE TABLE IF NOT EXISTS invoice_sequences (
    id INT PRIMARY KEY AUTO_INCREMENT,
    company_id INT NOT NULL DEFAULT 1,
    financial_year VARCHAR(10) NOT NULL,
    prefix VARCHAR(20) DEFAULT '',
    suffix VARCHAR(20) DEFAULT '',
    last_number INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_company_year (company_id, financial_year)
);

-- Settings table
CREATE TABLE IF NOT EXISTS settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Insert default company if not exists
INSERT IGNORE INTO companies (id, name, address, contact, email, gst_number, state, bank_name, bank_account, bank_ifsc) VALUES
(1, 'ELECTROFIX SOLUTION', 'D-171, 2nd Floor, Sector-10, Noida-201301', '8851369694', 'contact.electrofixsolution@gmail.com', '09DMFPP9986Q1ZJ', 'UTTAR PRADESH 09', 'HDFC Bank Ltd.', '50200086529101', 'HDFC0000088');

-- Insert sample customer if not exists
INSERT IGNORE INTO customers (id, name, billing_address, shipping_address, gstin, state, contact, email) VALUES
(1, 'BHAGWATI PRODUCTS LIMITED', 
 'Entire tower Tec-1,Entire tower Tec-2, Entire Tower WTC one C, TZ-13A, IT park, sector-techzone, Greater Noida, Gautambuddha Nagar, Uttar Pradesh, 201308',
 'Entire tower Tec-1,Entire tower Tec-2, Entire Tower WTC one C, TZ-13A, IT park, sector-techzone, Greater Noida, Gautambuddha Nagar, Uttar Pradesh, 201308',
 '09AACCB1828N1ZG', 'NOIDA 09', '9876543210', 'info@bhagwatiproducts.com');

-- Insert invoice sequence for current financial year if not exists
INSERT IGNORE INTO invoice_sequences (company_id, financial_year, prefix, suffix, last_number) VALUES
(1, '2025-26', '', '', 1);
