<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

include_once '../config/database.php';
include_once '../classes/Invoice.php';
include_once '../classes/Customer.php';
include_once '../classes/InvoiceItem.php';

$database = new Database();
$db = $database->getConnection();

$invoice = new Invoice($db);
$customer = new Customer($db);
$invoice_item = new InvoiceItem($db);

// Get posted data
$data = json_decode(file_get_contents("php://input"));

if (!empty($data->customer_name) && !empty($data->items)) {
    
    try {
        $db->beginTransaction();
        
        // Check if customer exists or create new
        $customer_stmt = $customer->search($data->customer_name);
        $existing_customer = $customer_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existing_customer) {
            $customer_id = $existing_customer['id'];
        } else {
            // Create new customer
            $customer->name = $data->customer_name;
            $customer->billing_address = $data->billing_address;
            $customer->shipping_address = $data->shipping_address;
            $customer->gstin = $data->customer_gstin;
            $customer->state = $data->customer_state;
            $customer->contact = $data->customer_contact ?? '';
            $customer->email = $data->customer_email ?? '';
            
            $customer_id = $customer->create();
        }
        
        // Calculate totals
        $subtotal = 0;
        foreach ($data->items as $item) {
            $subtotal += ($item->quantity * $item->rate);
        }
        
        $tax_rate = 0.18;
        $tax_amount = $subtotal * $tax_rate;
        $total_amount = $subtotal + $tax_amount;
        
        // Generate invoice number
        $invoice_number = $invoice->generateInvoiceNumber();
        
        // Create invoice
        $invoice->invoice_number = $invoice_number;
        $invoice->invoice_date = $data->invoice_date;
        $invoice->due_date = $data->due_date ?? null;
        $invoice->company_id = 1; // Default company
        $invoice->customer_id = $customer_id;
        $invoice->rgp_number = $data->rgp_number ?? '';
        $invoice->rgp_date = $data->rgp_date ?? null;
        $invoice->po_number = $data->po_number ?? '';
        $invoice->po_date = $data->po_date ?? null;
        $invoice->reverse_charge = $data->reverse_charge ?? 'NO';
        $invoice->tax_type = $data->tax_type ?? 'cgst_sgst';
        $invoice->subtotal = $subtotal;
        
        if ($data->tax_type == 'cgst_sgst') {
            $invoice->cgst_amount = $tax_amount / 2;
            $invoice->sgst_amount = $tax_amount / 2;
            $invoice->igst_amount = 0;
        } else {
            $invoice->cgst_amount = 0;
            $invoice->sgst_amount = 0;
            $invoice->igst_amount = $tax_amount;
        }
        
        $invoice->total_tax = $tax_amount;
        $invoice->total_amount = $total_amount;
        $invoice->amount_in_words = numberToWords($total_amount);
        $invoice->notes = $data->notes ?? '';
        $invoice->terms_conditions = 'Net 10 - Payment ten days after the invoice date';
        $invoice->status = 'draft';
        
        $invoice_id = $invoice->create();
        
        if ($invoice_id) {
            // Create invoice items
            foreach ($data->items as $item) {
                $invoice_item->invoice_id = $invoice_id;
                $invoice_item->product_id = null;
                $invoice_item->description = $item->description;
                $invoice_item->hsn_sac = $item->hsn_sac ?? '';
                $invoice_item->quantity = $item->quantity;
                $invoice_item->unit = $item->unit ?? 'PCS';
                $invoice_item->rate = $item->rate;
                $invoice_item->amount = $item->quantity * $item->rate;
                
                $invoice_item->create();
            }
            
            $db->commit();
            
            http_response_code(201);
            echo json_encode(array(
                "message" => "Invoice created successfully.",
                "invoice_id" => $invoice_id,
                "invoice_number" => $invoice_number
            ));
        } else {
            $db->rollback();
            http_response_code(503);
            echo json_encode(array("message" => "Unable to create invoice."));
        }
        
    } catch (Exception $e) {
        $db->rollback();
        http_response_code(503);
        echo json_encode(array("message" => "Error: " . $e->getMessage()));
    }
    
} else {
    http_response_code(400);
    echo json_encode(array("message" => "Unable to create invoice. Data is incomplete."));
}

// Number to words function
function numberToWords($number) {
    $ones = array(
        0 => 'ZERO', 1 => 'ONE', 2 => 'TWO', 3 => 'THREE', 4 => 'FOUR', 5 => 'FIVE',
        6 => 'SIX', 7 => 'SEVEN', 8 => 'EIGHT', 9 => 'NINE', 10 => 'TEN',
        11 => 'ELEVEN', 12 => 'TWELVE', 13 => 'THIRTEEN', 14 => 'FOURTEEN', 15 => 'FIFTEEN',
        16 => 'SIXTEEN', 17 => 'SEVENTEEN', 18 => 'EIGHTEEN', 19 => 'NINETEEN'
    );
    
    $tens = array(
        2 => 'TWENTY', 3 => 'THIRTY', 4 => 'FORTY', 5 => 'FIFTY',
        6 => 'SIXTY', 7 => 'SEVENTY', 8 => 'EIGHTY', 9 => 'NINETY'
    );
    
    if ($number < 20) {
        return $ones[$number];
    } elseif ($number < 100) {
        return $tens[intval($number / 10)] . ($number % 10 != 0 ? ' ' . $ones[$number % 10] : '');
    } elseif ($number < 1000) {
        return $ones[intval($number / 100)] . ' HUNDRED' . ($number % 100 != 0 ? ' ' . numberToWords($number % 100) : '');
    } elseif ($number < 100000) {
        return numberToWords(intval($number / 1000)) . ' THOUSAND' . ($number % 1000 != 0 ? ' ' . numberToWords($number % 1000) : '');
    } elseif ($number < 10000000) {
        return numberToWords(intval($number / 100000)) . ' LAKH' . ($number % 100000 != 0 ? ' ' . numberToWords($number % 100000) : '');
    } else {
        return numberToWords(intval($number / 10000000)) . ' CRORE' . ($number % 10000000 != 0 ? ' ' . numberToWords($number % 10000000) : '');
    }
}
?>
