<?php
require_once 'config/database.php';

class Invoice {
    private $conn;
    private $table_name = "invoices";

    public $id;
    public $invoice_number;
    public $invoice_date;
    public $due_date;
    public $company_id;
    public $customer_id;
    public $rgp_number;
    public $rgp_date;
    public $po_number;
    public $po_date;
    public $reverse_charge;
    public $tax_type;
    public $subtotal;
    public $cgst_amount;
    public $sgst_amount;
    public $igst_amount;
    public $total_tax;
    public $total_amount;
    public $amount_in_words;
    public $notes;
    public $terms_conditions;
    public $status;

    public function __construct($db) {
        $this->conn = $db;
    }

    // Generate next invoice number
    public function generateInvoiceNumber($company_id = 1) {
        $financial_year = $this->getCurrentFinancialYear();
        
        // Get or create sequence
        $query = "SELECT * FROM invoice_sequences WHERE company_id = ? AND financial_year = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $company_id);
        $stmt->bindParam(2, $financial_year);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $next_number = $row['last_number'] + 1;
            
            // Update sequence
            $update_query = "UPDATE invoice_sequences SET last_number = ? WHERE id = ?";
            $update_stmt = $this->conn->prepare($update_query);
            $update_stmt->bindParam(1, $next_number);
            $update_stmt->bindParam(2, $row['id']);
            $update_stmt->execute();
        } else {
            // Create new sequence
            $next_number = 1;
            $insert_query = "INSERT INTO invoice_sequences (company_id, financial_year, last_number) VALUES (?, ?, ?)";
            $insert_stmt = $this->conn->prepare($insert_query);
            $insert_stmt->bindParam(1, $company_id);
            $insert_stmt->bindParam(2, $financial_year);
            $insert_stmt->bindParam(3, $next_number);
            $insert_stmt->execute();
        }
        
        return str_pad($next_number, 3, '0', STR_PAD_LEFT) . '/' . $financial_year;
    }

    private function getCurrentFinancialYear() {
        $current_date = new DateTime();
        $current_year = $current_date->format('Y');
        $current_month = $current_date->format('n');
        
        if ($current_month >= 4) {
            return $current_year . '-' . substr($current_year + 1, 2);
        } else {
            return ($current_year - 1) . '-' . substr($current_year, 2);
        }
    }

    // Create invoice
    public function create() {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET invoice_number=:invoice_number, invoice_date=:invoice_date, due_date=:due_date,
                      company_id=:company_id, customer_id=:customer_id, rgp_number=:rgp_number,
                      rgp_date=:rgp_date, po_number=:po_number, po_date=:po_date,
                      reverse_charge=:reverse_charge, tax_type=:tax_type, subtotal=:subtotal,
                      cgst_amount=:cgst_amount, sgst_amount=:sgst_amount, igst_amount=:igst_amount,
                      total_tax=:total_tax, total_amount=:total_amount, amount_in_words=:amount_in_words,
                      notes=:notes, terms_conditions=:terms_conditions, status=:status";

        $stmt = $this->conn->prepare($query);

        // Sanitize
        $this->invoice_number = htmlspecialchars(strip_tags($this->invoice_number));
        $this->invoice_date = htmlspecialchars(strip_tags($this->invoice_date));
        $this->customer_id = htmlspecialchars(strip_tags($this->customer_id));

        // Bind values
        $stmt->bindParam(":invoice_number", $this->invoice_number);
        $stmt->bindParam(":invoice_date", $this->invoice_date);
        $stmt->bindParam(":due_date", $this->due_date);
        $stmt->bindParam(":company_id", $this->company_id);
        $stmt->bindParam(":customer_id", $this->customer_id);
        $stmt->bindParam(":rgp_number", $this->rgp_number);
        $stmt->bindParam(":rgp_date", $this->rgp_date);
        $stmt->bindParam(":po_number", $this->po_number);
        $stmt->bindParam(":po_date", $this->po_date);
        $stmt->bindParam(":reverse_charge", $this->reverse_charge);
        $stmt->bindParam(":tax_type", $this->tax_type);
        $stmt->bindParam(":subtotal", $this->subtotal);
        $stmt->bindParam(":cgst_amount", $this->cgst_amount);
        $stmt->bindParam(":sgst_amount", $this->sgst_amount);
        $stmt->bindParam(":igst_amount", $this->igst_amount);
        $stmt->bindParam(":total_tax", $this->total_tax);
        $stmt->bindParam(":total_amount", $this->total_amount);
        $stmt->bindParam(":amount_in_words", $this->amount_in_words);
        $stmt->bindParam(":notes", $this->notes);
        $stmt->bindParam(":terms_conditions", $this->terms_conditions);
        $stmt->bindParam(":status", $this->status);

        if ($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        return false;
    }

    // Read single invoice
    public function readOne() {
        $query = "SELECT i.*, c.name as customer_name, c.billing_address, c.shipping_address, 
                         c.gstin as customer_gstin, c.state as customer_state,
                         comp.name as company_name, comp.address as company_address,
                         comp.contact as company_contact, comp.email as company_email,
                         comp.gst_number as company_gst, comp.state as company_state,
                         comp.bank_name, comp.bank_account, comp.bank_ifsc
                  FROM " . $this->table_name . " i
                  LEFT JOIN customers c ON i.customer_id = c.id
                  LEFT JOIN companies comp ON i.company_id = comp.id
                  WHERE i.id = ? LIMIT 0,1";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();

        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $this->invoice_number = $row['invoice_number'];
            $this->invoice_date = $row['invoice_date'];
            $this->due_date = $row['due_date'];
            $this->company_id = $row['company_id'];
            $this->customer_id = $row['customer_id'];
            $this->rgp_number = $row['rgp_number'];
            $this->rgp_date = $row['rgp_date'];
            $this->po_number = $row['po_number'];
            $this->po_date = $row['po_date'];
            $this->reverse_charge = $row['reverse_charge'];
            $this->tax_type = $row['tax_type'];
            $this->subtotal = $row['subtotal'];
            $this->cgst_amount = $row['cgst_amount'];
            $this->sgst_amount = $row['sgst_amount'];
            $this->igst_amount = $row['igst_amount'];
            $this->total_tax = $row['total_tax'];
            $this->total_amount = $row['total_amount'];
            $this->amount_in_words = $row['amount_in_words'];
            $this->notes = $row['notes'];
            $this->terms_conditions = $row['terms_conditions'];
            $this->status = $row['status'];
            
            return $row;
        }
        return false;
    }

    // Read all invoices
    public function readAll($limit = 50, $offset = 0) {
        $query = "SELECT i.*, c.name as customer_name 
                  FROM " . $this->table_name . " i
                  LEFT JOIN customers c ON i.customer_id = c.id
                  ORDER BY i.created_at DESC
                  LIMIT ? OFFSET ?";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $limit, PDO::PARAM_INT);
        $stmt->bindParam(2, $offset, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt;
    }

    // Update invoice
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET invoice_date=:invoice_date, due_date=:due_date, customer_id=:customer_id,
                      rgp_number=:rgp_number, rgp_date=:rgp_date, po_number=:po_number,
                      po_date=:po_date, reverse_charge=:reverse_charge, tax_type=:tax_type,
                      subtotal=:subtotal, cgst_amount=:cgst_amount, sgst_amount=:sgst_amount,
                      igst_amount=:igst_amount, total_tax=:total_tax, total_amount=:total_amount,
                      amount_in_words=:amount_in_words, notes=:notes, terms_conditions=:terms_conditions,
                      status=:status
                  WHERE id=:id";

        $stmt = $this->conn->prepare($query);

        $stmt->bindParam(':invoice_date', $this->invoice_date);
        $stmt->bindParam(':due_date', $this->due_date);
        $stmt->bindParam(':customer_id', $this->customer_id);
        $stmt->bindParam(':rgp_number', $this->rgp_number);
        $stmt->bindParam(':rgp_date', $this->rgp_date);
        $stmt->bindParam(':po_number', $this->po_number);
        $stmt->bindParam(':po_date', $this->po_date);
        $stmt->bindParam(':reverse_charge', $this->reverse_charge);
        $stmt->bindParam(':tax_type', $this->tax_type);
        $stmt->bindParam(':subtotal', $this->subtotal);
        $stmt->bindParam(':cgst_amount', $this->cgst_amount);
        $stmt->bindParam(':sgst_amount', $this->sgst_amount);
        $stmt->bindParam(':igst_amount', $this->igst_amount);
        $stmt->bindParam(':total_tax', $this->total_tax);
        $stmt->bindParam(':total_amount', $this->total_amount);
        $stmt->bindParam(':amount_in_words', $this->amount_in_words);
        $stmt->bindParam(':notes', $this->notes);
        $stmt->bindParam(':terms_conditions', $this->terms_conditions);
        $stmt->bindParam(':status', $this->status);
        $stmt->bindParam(':id', $this->id);

        return $stmt->execute();
    }

    // Delete invoice
    public function delete() {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = ?";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        return $stmt->execute();
    }

    // Search invoices
    public function search($keywords) {
        $query = "SELECT i.*, c.name as customer_name 
                  FROM " . $this->table_name . " i
                  LEFT JOIN customers c ON i.customer_id = c.id
                  WHERE i.invoice_number LIKE ? OR c.name LIKE ?
                  ORDER BY i.created_at DESC";

        $stmt = $this->conn->prepare($query);
        $keywords = "%{$keywords}%";
        $stmt->bindParam(1, $keywords);
        $stmt->bindParam(2, $keywords);
        $stmt->execute();

        return $stmt;
    }
}
?>
